#!/usr/bin/env ruby
# $Id: starter_fallback.rb,v 1.2.2.1 2010/01/29 11:10:03 rjs Exp $
# Starter sample for fallback fonts
#
# Required software: PDFlib/PDFlib+PDI/PPS 8
# Required data: suitable fonts, Japanese CMaps
#

require './PDFlib'

# This is where the data files are. Adjust as necessary.
searchpath = "../data"
outfile = "starter_fallback.pdf"

llx = 50
lly = 50
urx = 800
ury = 550

headers = [ "Use case",
    "Option list for the 'fallbackfonts' option", "Base font",
    "With fallback font" ]

testcases = [
    # Add Euro glyph to an "encoding" which doesn't support it
    {
	"usecase" => "Extend 8-bit \"encoding\"",
	"fontname" => "Helvetica",
	"encoding" => "iso8859-1",
	"fallbackoptions" => "{fontname=Helvetica encoding=unicode " +
	    "forcechars=euro}",
	#
	# Reference Euro glyph by name (since it is missing from
	# the "encoding")
	#
	"text" => "123&euro;"},
    {
	"usecase" => "Use Euro glyph from another font",
	"fontname" => "Courier",
	"encoding" => "winansi",
	"fallbackoptions" => "{fontname=Helvetica encoding=unicode " +
	    "forcechars=euro textrise=-5%}",
	"text" => "123&euro;"},
    {
        "usecase" => "Enlarge all glyphs in a font",
        "fontname" => "Times-Italic",
	"encoding" => "winansi",
	#
	# Enlarge all glyphs to better match other fonts of the
	# same point size
	#
	"fallbackoptions" => "{fontname=Times-Italic encoding=unicode " +
	    "forcechars={U+0020-U+00FF} fontsize=120%}",
	"text" => "font size"},

    {
	"usecase" => "Add enlarged pictogram",
	"fontname" => "Times-Roman",
	"encoding" => "unicode",
	# pointing hand pictogram
	"fallbackoptions" => "{fontname=ZapfDingbats encoding=unicode " +
	    "forcechars=.a12 fontsize=150% textrise=-15%}",
	"text" => "Bullet symbol: &.a12;"},

    {
	"usecase" => "Add enlarged symbol glyph",
	"fontname" => "Times-Roman",
	"encoding" => "unicode",
	"fallbackoptions" => "{fontname=Symbol encoding=unicode " +
	    "forcechars=U+2663 fontsize=125%}",
	"text" => "Club symbol: &#x2663;"},
    #
    # Greek characters missing in the font will be pulled from Symbol
    # font
    #
    {
	"usecase" => "Add Greek characters to Latin font",
	"fontname" => "Times-Roman",
	"encoding" => "unicode",
	"fallbackoptions" => "{fontname=Symbol encoding=unicode}",
	"text" => "Greek text: &#x039B;&#x039F;&#x0393;&#x039F;&#x03A3;"},
	    
    # Font with end-user defined character {EUDC)
    {
	"usecase" => "Gaiji with EUDC font",
	"fontname" => "KozMinProVI-Regular",
	"encoding" => "unicode",
	"fallbackoptions" => "{fontname=EUDC encoding=unicode " +
	    "forcechars=U+E000 fontsize=140% textrise=-20%}",
	"text" => "Gaiji: &#xE000;"},

    # SING fontlet containing a single gaiji character
    {
	"usecase" => "Gaiji with SING font",
	"fontname" => "KozMinProVI-Regular",
	"encoding" => "unicode",
	"fallbackoptions" => "{fontname=PDFlibWing encoding=unicode " +
	    "forcechars=gaiji}",
	"text" => "Gaiji: &#xE000;"},

    {
	"usecase" => "Replace Latin characters in CJK font",
	"fontname" => "KozMinProVI-Regular",
	"encoding" => "unicode",
	"fallbackoptions" => "{fontname=Courier-Bold encoding=unicode " +
	    "forcechars={U+0020-U+007E}}",
	"text" => "Latin and &#x65E5;&#x672C;&#x8A9E;"},

    # Requires "Unicode BMP Fallback SIL" font in fallback.ttf
    # Identify missing glyphs caused by workflow problems
    {
	"usecase" => "Identify missing glyphs",
	"fontname" => "Times-Roman",
	"encoding" => "unicode",
	"fallbackoptions" => "{fontname=fallback encoding=unicode}",
	#
	# deliberately use characters which are not available in
	# the base font
	#
	"text" => "Missing glyphs: &#x1234; &#x672C; &#x8A9E;"}
]

begin
  p = PDFlib.new

  p.set_parameter("SearchPath", searchpath)
  p.set_parameter("charref", "true")
  p.set_parameter("textformat", "bytes");
  p.set_parameter("glyphcheck", "replace")

  #
  # This means that formatting and other errors will raise an
  # exception. This simplifies our sample code, but is not
  # recommended for production code.
  #
  p.set_parameter("errorpolicy", "exception")

  # Set an output path according to the name of the topic
  if (p.begin_document(outfile, "") == -1)
      raise "Error: " + p.get_errmsg()
  end

  p.set_info("Creator", "PDFlib starter sample")
  p.set_info("Title", "starter_fallback")

  # Start Page
  p.begin_page_ext(0, 0, "width=a4.height height=a4.width")

  table = -1
  row = 1

  # Table header
  1.step(headers.length) do |col|
    optlist = "fittextline={fontname=Helvetica-Bold encoding=unicode " +
	"fontsize=11} margin=4"
    table = p.add_table_cell(table, col, row, headers[col-1], optlist)
  end

  row = 2
  # Create fallback samples, one use case per row
  testcases.each do |testcase|
    col = 1

    # Column 1: description of the use case
    optlist = "fittextline={fontname=Helvetica encoding=unicode " +
	"fontsize=11} margin=4"
    table = p.add_table_cell(table, col, row, testcase["usecase"], optlist)
    col +=1

    # Column 2: reproduce option list literally
    optlist = "fittextline={fontname=Helvetica encoding=unicode " +
	"fontsize=10} margin=4"
    table = p.add_table_cell(table, col, row, testcase["fallbackoptions"], optlist)
    col +=1

    # Column 3: text with base font
    optlist = "fittextline={fontname=" + testcase["fontname"] +
	    " encoding=" + testcase["encoding"] +
	    " fontsize=11 replacementchar=? } margin=4"
    table = p.add_table_cell(table, col, row, testcase["text"], optlist)
    col +=1

    # Column 4: text with base font and fallback fonts
    optlist = "fittextline={fontname=" + testcase["fontname"] +
	    " encoding=" + testcase["encoding"] +
	    " fontsize=11 fallbackfonts={" +
	    testcase["fallbackoptions"] + "}} margin=4"
    table = p.add_table_cell(table, col, row, testcase["text"], optlist)
    col +=1
    row +=1
  end

  # Place the table
  optlist = "header=1 fill={{area=rowodd " +
	  "fillcolor={gray 0.9}}} stroke={{line=other}} "
  result = p.fit_table(table, llx, lly, urx, ury, optlist)

  if (result == "_error")
      raise "Couldn't place table: " + p.get_errmsg()
  end

  p.end_page_ext("")
  p.end_document("")

rescue  PDFlibException => pe
  print "PDFlib exception occurred:\n"
  print "[" + pe.get_errnum.to_s + "] " + pe.get_apiname + \
                  ": " + pe.get_errmsg + "\n"
rescue  Exception => e
  print e.backtrace.join("\n") + "\n" + e.to_s
ensure
  p.delete() if p
end

# vim: sw=2
