#!/usr/bin/env ruby
# $Id: starter_geospatial.rb,v 1.2 2009/09/22 18:07:44 rjs Exp $
# Starter for georeferenced PDF:
# Import an image with a map and add geospatial reference information
#
# Sample map and coordinates:
# We use a map from www.openstreetmap.com; the geospatial coordinates of the
# image edges were also provided by that Web site.
# The coordinate system is WGS84 which is also used for GPS.
#
# Required software: PDFlib/PDFlib+PDI/PPS 8
# Required data: image file and associated geospatial reference information
#/

require './PDFlib'

# This is where the data files are. Adjust if necessary.
searchpath = "../data"
outfile = "starter_geospatial.pdf"

imagefile = "munich.png"

# WKT for plain latitude/longitude values in WGS84
georef = "worldsystem={type=geographic wkt={" +
    "GEOGCS[\"WGS 84\"," +
    "  DATUM[\"WGS_1984\", SPHEROID[\"WGS 84\", 6378137,298.257223563]]," +
    "  PRIMEM[\"Greenwich\", 0.0]," +
    "  UNIT[\"Degree\", 0.01745329251994328]]" +
    "}} linearunit=M areaunit=SQM angularunit=degree"

# world coordinates of the image (in degrees)
worldpoints = [
    48.145, # latitude of top edge
    11.565, # longitude of left edge
    11.59,  # longitude of right edge
    48.13   # latitude of bottom edge
]

begin
  p = PDFlib.new

  p.set_parameter("SearchPath", searchpath)

  # This means we must check return values of load_font() etc.
  p.set_parameter("errorpolicy", "return")

  # Start the document
  if (p.begin_document(outfile, "compatibility=1.7ext3") == -1)
      raise "Error: " + p.get_errmsg()
  end

  p.set_info("Creator", "PDFlib starter sample")
  p.set_info("Title", "starter_geospatial")

  # Generate georeference option list
  # Use the four corners as reference points; (0,0)=lower left etc.
  optlist = "georeference={" + georef + " mappoints={0 0  1 0  1 1  0 1} "
  optlist += "worldpoints={"

  # lower left corner
  optlist += worldpoints[3].to_s + " " + worldpoints[1].to_s + " "
  # lower right corner
  optlist += worldpoints[3].to_s + " " + worldpoints[2].to_s + " "
  # upper right corner
  optlist += worldpoints[0].to_s + " " + worldpoints[2].to_s + " "
  # upper left corner
  optlist += worldpoints[0].to_s + " " + worldpoints[1].to_s

  optlist += "} }"

  # Load the image with geospatial reference attached
  image = p.load_image("auto", imagefile, optlist)
  if (image == -1)
      raise "Error: " + p.get_errmsg()
  end

  p.begin_page_ext(0, 0, "width=a4.width height=a4.height")

  # Create caption
  p.fit_textline("Map with geospatial reference information", 50, 700,
      "fontname=LuciduxSans-Oblique encoding=winansi fontsize=18")

  # Place the map on the page
  p.fit_image(image, 50, 50, "boxsize={500 600} fitmethod=meet")

  p.end_page_ext("")

  p.end_document("")

rescue  PDFlibException => pe
  print "PDFlib exception occurred:\n"
  print "[" + pe.get_errnum.to_s + "] " + pe.get_apiname + \
                  ": " + pe.get_errmsg + "\n"
rescue  Exception => e
  print e.backtrace.join("\n") + "\n" + e.to_s
ensure
  p.delete() if p
end

# vim: sw=2
