#!/usr/bin/env ruby
# $Id: starter_opentype.rb,v 1.2 2009/09/22 18:07:44 rjs Exp $
# Starter sample for OpenType font features
#
# Demonstrate various typographic OpenType features after checking
# whether a particular feature is supported in a font.
#
# Required software: PDFlib/PDFlib+PDI/PPS 8
# Required data: suitable fonts with OpenType feature tables
#
# This sample uses a default font which includes a few features.
# For better results you should replace the default font with a suitable
# commercial font. Depending on the implementation of the features you
# may also have to replace the sample text below.
#
# Some ideas for suitable test fonts:
# Palatino Linotype: standard Windows font with many OpenType features
#/

require './PDFlib'

# This is where the data files are. Adjust as necessary.
searchpath = "../data"
outfile = "starter_opentype.pdf"

llx = 50
lly = 50
urx = 800
ury = 550

# This font will be used unless another one is specified in the table
defaulttestfont = "DejaVuSerif"

headers = [
    "OpenType feature",
    "Option list",
    "Font name",
    "Raw input (feature disabled)",
    "Feature enabled"
]

testcases = [
{
  "description" => "ligatures",
  "fontname" => "",
  "feature" => "liga",
  "text" => "ff fi fl ffi ffl"
},
{
  "description" => "discretionary ligatures",
  "fontname" => "",
  "feature" => "dlig",
  "text" => "st c/o"
},
{
  "description" => "historical ligatures",
  "fontname" => "",
  "feature" => "hlig",
  "text" => "&.longs;b &.longs;t"
},
{
  "description" => "small capitals",
  "fontname" => "",
  "feature" => "smcp",
  "text" => "PostScript"
},
{
  "description" => "ordinals",
  "fontname" => "",
  "feature" => "ordn",
  "text" => "1o 2a 3o"
},
{
  "description" => "fractions",
  "fontname" => "",
  "feature" => "frac",
  "text" => "1/2 1/4 3/4"
},
{
  "description" => "alternate fractions",
  "fontname" => "",
  "feature" => "afrc",
  "text" => "1/2 1/4 3/4"
},
{
  "description" => "slashed zero",
  "fontname" => "",
  "feature" => "zero",
  "text" => "0"
},
{
  "description" => "historical forms",
  "fontname" => "",
  "feature" => "hist",
  "text" => "s"
},
{
  "description" => "proportional figures",
  "fontname" => "",
  "feature" => "pnum",
  "text" => "0123456789"
},
{
  "description" => "old-style figures",
  "fontname" => "",
  "feature" => "onum",
  "text" => "0123456789"
},
{
  "description" => "lining figures",
  "fontname" => "",
  "feature" => "lnum",
  "text" => "0123456789"
},
{
  "description" => "superscript",
  "fontname" => "",
  "feature" => "sups",
  "text" => "0123456789"
}
]

begin
  p = PDFlib.new

  p.set_parameter("SearchPath", searchpath)
  p.set_parameter("textformat", "bytes")
  p.set_parameter("charref", "true")

  # This means that formatting and other errors will raise an
  # exception. This simplifies our sample code, but is not
  # recommended for production code.
  #/
  p.set_parameter("errorpolicy", "exception")

  # Set an output path according to the name of the topic
  if (p.begin_document(outfile, "") == -1)
      raise "Error: " + p.get_errmsg()
  end

  p.set_info("Creator", "PDFlib starter sample")
  p.set_info("Title", "starter_opentype")

  # Start Page
  p.begin_page_ext(0, 0, "width=a4.height height=a4.width")

  table = -1
  row = 1
  col = 1

  # Table header
  headers.each do |header|
    optlist = "fittextline={fontname=Helvetica-Bold encoding=unicode " +
       "fontsize=12} margin=4"
    table = p.add_table_cell(table, col, row, header, optlist)
    col += 1
  end

  # Create a table with feature samples, one feature per table row
  row = 2
  testcases.each do |test|
    # Use the entry in the test table if available, and the
    # default test font otherwise. This way we can easily check
    # a font for all features, as well as insert suitable fonts
    # for individual features.
    #/
    if (test["fontname"] != "")
	testfont = test["fontname"]
    else
	testfont = defaulttestfont
    end

    col=1

    # Common option list for columns 1-3
    optlist = "fittextline={fontname=Helvetica encoding=unicode " +
	"fontsize=12} margin=4"

    # Column 1: feature description
    table = p.add_table_cell(table, col, row, test["description"], optlist)
    col += 1

    # Column 2: option list
    buf = "features={" + test["feature"] + "}"
    table = p.add_table_cell(table, col, row, buf, optlist)
    col += 1

    # Column 3: font name
    table = p.add_table_cell(table, col, row, testfont, optlist)
    col += 1

    # Column 4: raw input text with  feature disabled
    optlist = "fittextline={fontname={" + testfont + "} encoding=unicode " +
	 "fontsize=12 embedding} margin=4"
    table = p.add_table_cell(table, col, row, test["text"], optlist)
    col += 1

    # Column 5: text with enabled feature, or warning if the
    # feature is not available in the font
    #/
    font = p.load_font(testfont, "unicode", "embedding")

    # Check whether font contains the required feature table
    optlist = "name=" + test["feature"]
    if (p.info_font(font, "feature", optlist) == 1)
	# feature is available: apply it to the text
	optlist = "fittextline={fontname={" + testfont + "} encoding=unicode " +
	    "fontsize=12 embedding features={" + test["feature"] + "}} margin=4"
	table = p.add_table_cell(table, col, row, test["text"], optlist)
    else
	# feature is not available: emit a warning
	optlist = "fittextline={fontname=Helvetica encoding=unicode " +
	     "fontsize=12 fillcolor=red} margin=4"
	table = p.add_table_cell(table, col, row,
		"(feature not available in this font)", optlist)
    end
    row += 1
  end

  # Place the table
  optlist = "header=1 fill={{area=rowodd fillcolor={gray 0.9}}} " +
      "stroke={{line=other}} "
  result = p.fit_table(table, llx, lly, urx, ury, optlist)

  if (result == "_error")
      raise "Error: " + p.get_errmsg()
  end

  p.end_page_ext("")
  p.end_document("")

rescue  PDFlibException => pe
  print "PDFlib exception occurred:\n"
  print "[" + pe.get_errnum.to_s + "] " + pe.get_apiname + \
                  ": " + pe.get_errmsg + "\n"
rescue  Exception => e
  print e.backtrace.join("\n") + "\n" + e.to_s
ensure
  p.delete() if p
end

# vim: sw=2
