#!/usr/bin/env ruby
# $Id: starter_portfolio.rb,v 1.2 2009/09/22 18:07:44 rjs Exp $
#
# PDF portfolio starter:
# Package multiple PDF and other documents into a PDF portfolio
# The generated PDF portfolio requires Acrobat 9 for proper
# viewing. The documents in the Portfolio will be assigned predefined
# and custom metadata fields; for the custom fields a schema description
# is created.
#
# Acrobat 8 will only display a "PDF package" with a flat list of documents
# without any folder structure.
#
# Required software: PDFlib/PDFlib+PDI/PPS 8
# Required data: PDF and other input documents
#

require './PDFlib'

# This is where the data files are. Adjust as necessary.
searchpath = "../data"

# The documents for the Portfolio along with description and metadata
pfroot = [
{
    "filename" => "TIR_____.AFM",
    "description" => "Metrics for Times-Roman",
    "status" => "internal",
    "id" => 200
},
{
    "filename" => "nesrin.jpg",
    "description" => "Zabrisky point",
    "status" => "archived",
    "id" => 300
}
]
pffolder = [
{
    "filename" => "PDFlib-real-world.pdf",
    "description" => "PDFlib in the real world",
    "status" => "published",
    "id" => 100
},
{
    "filename" => "PDFlib-datasheet.pdf",
    "description" => "Generate PDF on the fly",
    "status" => "published",
    "id" => 101
},
{
    "filename" => "TET-datasheet.pdf",
    "description" => "Extract text and images from PDF",
    "status" => "published",
    "id" => 102
},
{
    "filename" => "PLOP-datasheet.pdf",
    "description" => "PDF Linearization, Optimization, Protection",
    "status" => "published",
    "id" => 103
},
{
    "filename" => "pCOS-datasheet.pdf",
    "description" => "PDF Information Retrieval Tool",
    "status" => "published",
    "id" => 104
}
]

begin
  # create a new PDFlib object
  p = PDFlib.new

  p.set_parameter("SearchPath", searchpath)

  # This means we must check return values of load_font() etc.
  p.set_parameter("errorpolicy", "return")

  if (p.begin_document("starter_portfolio.pdf", "compatibility=1.7ext3") == -1)
      raise "Error: " + p.get_errmsg()
  end

  p.set_info("Creator", "PDFlib starter sample")
  p.set_info("Title", "starter_portfolio")

  # Insert all files for the root folder along with their description
  # and the following custom fields:
  # status   string describing the document status
  # id       numerical identifier, prefixed with "PHX"
  #
  pfroot.each do |pf|
      optlist = "description={" + pf["description"] + "} " +
	  "fieldlist={ {key=status value=" + pf["status"] + "} " +
	  "{key=id value=" + pf["id"].to_s + " prefix=PHX type=text} }"

      # -1 means root folder
      p.add_portfolio_file(-1, pf["filename"], optlist)
  end

  # Create the "datasheets" folder in the root folder
  folder = p.add_portfolio_folder(-1, "datasheets", "")

  # Insert documents in the "datasheets" folder along with
  # description and custom fields
  #
  pffolder.each do |pf|
      optlist = "description={" + pf["description"] + "} " +
	  "fieldlist={ {key=status value=" + pf["status"] + "} " +
	  "{key=id value=" + pf["id"].to_s + " prefix=PHX type=text} }"

      # Add the file to the "datasheets" folder
      p.add_portfolio_file(folder, pf["filename"], optlist)
  end

  # Create a single-page document as cover sheet
  p.begin_page_ext(0, 0, "width=a4.width height=a4.height")

  font = p.load_font("Helvetica", "winansi", "")
  if (font == -1)
      raise "Error: " + p.get_errmsg()
  end

  p.setfont(font, 24)
  p.fit_textline("Welcome to the PDFlib Portfolio sample!", 50, 700, "")

  p.end_page_ext("")

  # Set options for Portfolio display
  optlist = "portfolio={initialview=detail "

  # Add schema definition for Portfolio metadata
  optlist += "schema={ " +
  # Some predefined fields are included here to make them visible.
  "{order=1 label=Name key=_filename visible editable} " +
  "{order=2 label=Description key=_description visible} " +
  "{order=3 label=Size key=_size visible} " +
  "{order=4 label={Last edited} key=_moddate visible} "

  # User-defined fields
  optlist +=
  "{order=5 label=Status key=status type=text editable} " +
  "{order=6 label=ID key=id type=text editable} "

  optlist += "}}"

  p.end_document(optlist)

rescue  PDFlibException => pe
  print "PDFlib exception occurred:\n"
  print "[" + pe.get_errnum.to_s + "] " + pe.get_apiname + \
                  ": " + pe.get_errmsg + "\n"
rescue  Exception => e
  print e.backtrace.join("\n") + "\n" + e.to_s
ensure
  p.delete() if p
end

# vim: sw=2
