#!/bin/sh
# $Id: starter_block.tcl,v 1.6 2008/07/29 11:19:30 rjs Exp $
#
# Hide the exec to Tcl but not to the shell by appending a backslash\
exec tclsh "$0" ${1+"$@"}

# Starter for georeferenced PDF:
# Import an image with a map and add geospatial reference information
#
# Sample map and coordinates:
# We use a map from www.openstreetmaPDF_com; the geospatial coordinates of the
# image edges were also provided by that Web site.
# The coordinate system is WGS84 which is also used for GPS.
#
# Required software: PDFlib/PDFlib+PDI/PPS 8
# Required data: image file and associated geospatial reference information
#

# The linsert line is unnecessary if PDFlib has been installed
# in the Tcl package directory
set auto_path [linsert $auto_path 0 .libs .]

package require pdflib 8.0

# This is where the data files are. Adjust if necessary.
set searchpath "../data"
set outfile "starter_geospatial.pdf"

set imagefile "munich.png"

# WKT for plain latitude/longitude values in WGS84
set georef \
"worldsystem={type=geographic wkt={\
GEOGCS\[\"WGS 84\",\
  DATUM\[\"WGS_1984\", SPHEROID\[\"WGS 84\", 6378137,298.257223563\]\],\
  PRIMEM\[\"Greenwich\", 0.0\],\
  UNIT\[\"Degree\", 0.01745329251994328\]\]\
}} linearunit=M areaunit=SQM angularunit=degree"

# world coordinates of the image (in degrees)
set worldpoints(0) 48.145
set worldpoints(1) 11.565
set worldpoints(2) 11.59
set worldpoints(3) 48.13

# create a new PDFlib object
set p [PDF_new]

if { [catch {

    PDF_set_parameter $p "SearchPath" $searchpath

    # This means we must check return values of load_font() etc.
    PDF_set_parameter $p "errorpolicy" "return"

    # Start the document
    if {[PDF_begin_document $p $outfile "compatibility=1.7ext3"] == -1} {
	puts stderr [format "Error: %s\n" [PDF_get_errmsg $p]]
	exit
    }

    PDF_set_info $p "Creator" "PDFlib starter sample"
    PDF_set_info $p "Title" "starter_geospatial"

    # Generate georeference option list
    # Use the four corners as reference points; (0,0)=lower left etc.
    set georefoptlist [format "georeference={%s mappoints={0 0  1 0  1 1  0 1} worldpoints={" $georef ]

    # lower left corner
    append georefoptlist [format "%g %g " $worldpoints(3) $worldpoints(1)]
    # lower right corner
    append georefoptlist [format "%g %g " $worldpoints(3) $worldpoints(2)]
    # upper right corner
    append georefoptlist [format "%g %g " $worldpoints(0) $worldpoints(2)]
    # upper left corner
    append georefoptlist [format "%g %g " $worldpoints(0) $worldpoints(1)]

    append georefoptlist "} }"

    # Load the image with geospatial reference attached
    set image [PDF_load_image $p "auto" $imagefile $georefoptlist]
    if {$image == -1} {
	puts stderr [format "Error: %s\n" [PDF_get_errmsg $p]]
	exit
    }

    PDF_begin_page_ext $p 0 0 "width=a4.width height=a4.height"

    # Create caption
    PDF_fit_textline $p "Map with geospatial reference information" \
	50 700 "fontname=LuciduxSans-Oblique encoding=winansi fontsize=18"

    # Place the map on the page
    PDF_fit_image $p $image 50 50 "boxsize={500 600} fitmethod=meet"

    PDF_end_page_ext $p ""
    PDF_end_document $p ""
} result] } {
    puts stderr "$result"
}

# delete the PDFlib object
PDF_delete $p
