#!/bin/sh
# $Id: starter_block.tcl,v 1.6 2008/07/29 11:19:30 rjs Exp $
#
# Hide the exec to Tcl but not to the shell by appending a backslash\
exec tclsh "$0" ${1+"$@"}

# $Id: starter_opentype.pl,v 1.1 2009/09/11 10:32:16 stm Exp $
# Starter sample for OpenType font features
#
# Demonstrate various typographic OpenType features after checking
# whether a particular feature is supported in a font.
#
# Required software: PDFlib/PDFlib+PDI/PPS 8
# Required data: suitable fonts with OpenType feature tables
#
# This sample uses a default font which includes a few features.
# For better results you should replace the default font with a suitable
# commercial font. Depending on the implementation of the features you
# may also have to replace the sample text below.
#
# Some ideas for suitable test fonts:
# Palatino Linotype: standard Windows font with many OpenType features

# The linsert line is unnecessary if PDFlib has been installed
# in the Tcl package directory
set auto_path [linsert $auto_path 0 .libs .]

package require pdflib 8.0


# This is where the data files are. Adjust as necessary.
set searchpath "../data";
set outfile "starter_opentype.pdf";

set llx 50.0
set lly 50.0
set urx 800.0
set ury 550.0

# This font will be used unless another one is specified in the table 
set defaulttestfont "DejaVuSerif";

set headers(0) "OpenType feature"
set headers(1) "Option list"
set headers(2) "Font name"
set headers(3) "Raw input (feature disabled)"
set headers(4) "Feature enabled"

set MAXCOL 5
# struct testcase
set description 0
set fontname 1
set feature 2
set text 3

set testcases(0,0) "ligatures"
set testcases(0,1) ""
set testcases(0,2) "liga"
set testcases(0,3) "ff fi fl ffi ffl"

set testcases(1,0) "discretionary ligatures"
set testcases(1,1) ""
set testcases(1,2) "dlig"
set testcases(1,3) "st c/o"

set testcases(2,0) "historical ligatures"
set testcases(2,1) ""
set testcases(2,2) "hlig"
set testcases(2,3) "&.longs;b &.longs;t"

set testcases(3,0) "small capitals"
set testcases(3,1) ""
set testcases(3,2) "smcp"
set testcases(3,3) "PostScript"

set testcases(4,0) "ordinals"
set testcases(4,1) ""
set testcases(4,2) "ordn"
set testcases(4,3) "1o 2a 3o"

set testcases(5,0) "fractions"
set testcases(5,1) ""
set testcases(5,2) "frac"
set testcases(5,3) "1/2 1/4 3/4"

set testcases(6,0) "alternate fractions"
set testcases(6,1) ""
set testcases(6,2) "afrc"
set testcases(6,3) "1/2 1/4 3/4"

set testcases(7,0) "slashed zero"
set testcases(7,1) ""
set testcases(7,2) "zero"
set testcases(7,3) "0"

set testcases(8,0) "historical forms"
set testcases(8,1) ""
set testcases(8,2) "hist"
set testcases(8,3) "s"

set testcases(9,0) "proportional figures"
set testcases(9,1) ""
set testcases(9,2) "pnum"
set testcases(9,3) "0123456789"

set testcases(10,0) "old-style figures"
set testcases(10,1) ""
set testcases(10,2) "onum"
set testcases(10,3) "0123456789"

set testcases(11,0) "lining figures"
set testcases(11,1) ""
set testcases(11,2) "lnum"
set testcases(11,3) "0123456789"

set testcases(12,0) "superscript"
set testcases(12,1) ""
set testcases(12,2) "sups"
set testcases(12,3) "0123456789"

set n_testcases 13

set p [PDF_new]

if { [catch {

    PDF_set_parameter $p "SearchPath" $searchpath
    PDF_set_parameter $p "charref" "true"

    # This means that formatting and other errors will raise an
    # exception. This simplifies our sample code, but is not
    # recommended for production code.
    PDF_set_parameter $p "errorpolicy" "exception"

    # Set an output path according to the name of the topic
    if {[PDF_begin_document $p $outfile ""] == -1} {
	puts stderr [format "Error: %s\n" [PDF_get_errmsg $p]]
	exit
    }

    PDF_set_info $p "Creator" "PDFlib starter sample"
    PDF_set_info $p "Title" "starter_opentype"

    # Start Page
    PDF_begin_page_ext $p 0 0 "width=a4.height height=a4.width"

    set table -1
    
    # Table header
    set col 1
    set row 1
    for {set col 1} {$col <= $MAXCOL} {incr col} {
        set optlist "fittextline={fontname=Helvetica-Bold encoding=unicode \
fontsize=12} margin=4"
        set table [PDF_add_table_cell $p $table $col $row \
				$headers([expr $col-1]) $optlist]
    }

    incr row
    
    # Create a table with feature samples, one feature per table row
    for {set test 0} {$test < $n_testcases} {incr test} {
        # Use the entry in the test table if available, and the
        # default test font otherwise. This way we can easily check
        # a font for all features, as well as insert suitable fonts
        # for individual features.
        if {$testcases($test,$fontname) != ""} {
            set testfont $testcases($test,$fontname)
        } else {
            set testfont $defaulttestfont
	}

        set col 1

        # Common option list for columns 1-3
        set optlist "fittextline={fontname=Helvetica encoding=unicode \
fontsize=12} margin=4"
        

        # Column 1: feature description
        set table [PDF_add_table_cell $p $table $col $row \
			$testcases($test,$description) $optlist]

        incr col
        
        # Column 2: option list
        set buf [format "features={%s}" $testcases($test,$feature)]
        set table [PDF_add_table_cell $p $table $col $row $buf $optlist]

        incr col
        
        # Column 3: font name
        set table [PDF_add_table_cell $p $table $col $row $testfont $optlist]

        incr col
        
        # Column 4: raw input text with  feature disabled
        set optlist [format "fittextline={fontname={%s} encoding=unicode \
fontsize=12 embedding} margin=4" $testfont]
        
        set table [PDF_add_table_cell $p $table $col $row $$testcases($test,$text) $optlist]

        incr col
        
        # Column 5: text with enabled feature, or warning if the
        # feature is not available in the font
        set font [PDF_load_font $p $testfont "unicode" "embedding"]

        # Check whether font contains the required feature table
        set optlist [format "name=%s" $testcases($test,$feature)]
        if {[PDF_info_font $p $font "feature" $optlist] == 1} {
            # feature is available: apply it to the text
            set optlist [format "fittextline={fontname={%s} encoding=unicode \
fontsize=12 embedding features={%s}} margin=4" \
                 $testfont $testcases($test,$feature)]
            
            set table [PDF_add_table_cell $p $table $col $row $testcases($test,$text) $optlist]
	} else {
            # feature is not available: emit a warning
            set optlist "fittextline={fontname=Helvetica encoding=unicode \
fontsize=12 fillcolor=red} margin=4"
            
            set table [PDF_add_table_cell $p $table $col $row \
                    "(feature not available in this font)" $optlist]
	}
        
        incr row
    }

    # Place the table
    set optlist "header=1 fill={{area=rowodd fillcolor={gray 0.9}}} \
stroke={{line=other}}"
    set result [PDF_fit_table $p $table $llx $lly $urx $ury $optlist]

    if {$result == "_error"} {
	puts stderr [format "Error: %s\n" [PDF_get_errmsg $p]]
	exit
    }

    PDF_end_page_ext $p ""
    PDF_end_document $p ""
} result] } {
    puts stderr "$result"
}

# delete the PDFlib object
PDF_delete $p
