#!/bin/sh
# $Id: starter_pdfx4.tcl,v 1.2 2009/09/17 12:31:27 rjs Exp $
#
# Hide the exec to Tcl but not to the shell by appending a backslash\
exec tclsh "$0" ${1+"$@"}

# PDF/X-4 starter:
# Create PDF/X-4 conforming output with layer variants and transparency
#
# A low-level layer is created for each of several languages, as well
# as an image layer. Each of the language layers together with the
# image layer forms a "layer variant" according to PDF/X-4 (in Acrobat
# layer variants are called "configurations").
# This ensures that low-level layers cannot be enabled/disabled individually,
# but only via the corresponding layer variant. This prevents accidental
# printing of a language layer without the required image layer.
#
# The document contains transparent text which is allowed in
# PDF/X-4, but not earlier PDF/X standards.
#
# Required software: PDFlib/PDFlib+PDI/PPS 8
# Required data: font file, image file, ICC output intent profile
#                (see www.pdflib.com for ICC profiles)

# The linsert line is unnecessary if PDFlib has been installed
# in the Tcl package directory
set auto_path [linsert $auto_path 0 .libs .]

package require pdflib 8.0

# This is where the data files are. Adjust as necessary.
set searchpath "../data"
set imagefile "zebra.tif"

set p [PDF_new]

if { [catch {
    # This means we must check return values of load_font  etc.
    PDF_set_parameter $p "errorpolicy" "return"

    PDF_set_parameter $p "SearchPath" $searchpath

    if {[PDF_begin_document $p "starter_pdfx4.pdf" "pdfx=PDF/X-4"] \
            == -1} {
	puts stderr [format "Error: %s\n" [PDF_get_errmsg $p]]
	PDF_delete $p
	exit
    }

    PDF_set_info $p "Creator" "PDFlib starter sample"
    PDF_set_info $p "Title" "starter_pdfx4"

    if {[PDF_load_iccprofile $p "ISOcoated.icc" "usage=outputintent"] == -1} {
	puts stderr [format "Error: %s\n" [PDF_get_errmsg $p]]
        puts stderr "Please install the ICC profile package from \
www.pdflib.com to run the PDF/X-4 starter sample.\n"
	PDF_delete $p
	exit
    }

    # Define the low-level layers. These cannot be controlled directly
    # in Acrobat's layer pane.

    set layer_english [PDF_define_layer $p "English text" ""]
    set layer_german  [PDF_define_layer $p "German text" ""]
    set layer_french  [PDF_define_layer $p "French text" ""]
    set layer_image   [PDF_define_layer $p "Images" ""]

    # Define the layer combinations for document variants. The variants
    # control the low-level layers, and can be activated in Acrobat's
    # layer pane. Using layer variants we can make sure that the image
    # layer cannot accidentally be disabled; it will always accompany
    # the text regardless of the selected language.

    # Define a radio button relationship for the language layers.
    # Individual layers will only be visible in Acrobat X (but
    # not Acrobat 9).
    #
    set optlist [format "group={%d %d %d}" \
	    $layer_english $layer_german $layer_french]
    PDF_set_layer_dependency $p "Radiobtn" $optlist

    set optlist [format "variantname={English variant} includelayers={%d %d} \
defaultvariant=true" $layer_english $layer_image]
    PDF_set_layer_dependency $p "Variant" $optlist

    set optlist [format "variantname={German variant} includelayers={%d %d}" \
    	$layer_german $layer_image]
    PDF_set_layer_dependency $p "Variant" $optlist

    set optlist [format "variantname={French variant} includelayers={%d %d}" \
    	$layer_french $layer_image]
    PDF_set_layer_dependency $p "Variant" $optlist

    PDF_begin_page_ext $p 595 842 ""

    # Font embedding is required for PDF/X
    set font [PDF_load_font $p "LuciduxSans-Oblique" "winansi" "embedding"]

    if {$font == -1} {
	puts stderr [format "Error: %s\n" [PDF_get_errmsg $p]]
	PDF_delete $p
	exit
    }

    PDF_setfont $p $font 24

    PDF_begin_layer $p $layer_english

    PDF_fit_textline $p "PDF/X-4 starter sample with layers" 50 700 ""

    PDF_begin_layer $p $layer_german
    PDF_fit_textline $p "PDF/X-4 Starter-Beispiel mit Ebenen" 50 700 ""

    PDF_begin_layer $p $layer_french
    PDF_fit_textline $p "PDF/X-4 Starter exemple avec des calques" 50 700 ""

    PDF_begin_layer $p $layer_image

    PDF_setfont $p $font 48

    # The RGB image needs an ICC profile; we use sRGB.
    set icc [PDF_load_iccprofile $p "sRGB" ""]
    set optlist [format "iccprofile=%d" $icc]
    set image [PDF_load_image $p "auto" $imagefile $optlist]

    if {$image == -1} {
	puts stderr [format "Error: %s\n" [PDF_get_errmsg $p]]
	PDF_delete $p
	exit
    }

    # Place a diagonal stamp across the image area
    set width [PDF_info_image $p $image "width" ""]
    set height [PDF_info_image $p $image "height" ""]

    set optlist [format "boxsize={%f %f} stamp=ll2ur" $width $height]
    PDF_fit_textline $p "Zebra" 0 0 $optlist

    # Set transparency in the graphics state
    set gstate [PDF_create_gstate $p "opacityfill=0.5"]
    PDF_set_gstate $p $gstate

    # Place the image on the page and close it
    PDF_fit_image $p $image 0.0 0.0 ""
    PDF_close_image $p $image

    # Close all layers
    PDF_end_layer $p

    PDF_end_page_ext $p ""

    PDF_end_document $p ""

} result] } {
    puts stderr "$result"
}

# delete the PDFlib object
PDF_delete $p

