#!/bin/sh
# $Id: starter_pdfx5g.tcl,v 1.2 2009/09/17 12:31:27 rjs Exp $
#
# Hide the exec to Tcl but not to the shell by appending a backslash\
exec tclsh "$0" ${1+"$@"}

# PDF/X-5g starter:
# Create PDF/X-5g conforming output with a reference to an external page
#
# The external document from which a page is referenced must conform to
# one of the following standards:
# PDF/X-1a:2003, PDF/X-3:2002, PDF/X-4, PDF/X-4p, PDF/X-5g, or PDF/X-5pg
#
# In order to properly display and print the referenced target page with
# Acrobat you must configure Acrobat appropriately (see PDFlib Tutorial),
# and the target PDF must be available to Acrobat.
#
# Required software: PDFlib/PDFlib+PDI/PPS 8
# Required data: font file, external PDF/X target, ICC output intent profile
#                (see www.pdflib.com for ICC profiles)
 

# The linsert line is unnecessary if PDFlib has been installed
# in the Tcl package directory
set auto_path [linsert $auto_path 0 .libs .]

package require pdflib 8.0

# This is where the data files are. Adjust as necessary.
set searchpath "../data"
set targetname "x5target.pdf"

set linewidth 2.0

set p [PDF_new]

if { [catch {
    # This means we must check return values of load_font() etc.
    PDF_set_parameter $p "errorpolicy" "return"

    PDF_set_parameter $p "SearchPath" $searchpath

    if {[PDF_begin_document $p "starter_pdfx5g.pdf" "pdfx=PDF/X-5g"] == -1} {
	puts stderr [format "Error: %s\n" [PDF_get_errmsg $p]]
	PDF_delete $p
	exit
    }

    PDF_set_info $p "Creator" "PDFlib starter sample"
    PDF_set_info $p "Title" "starter_pdfx5g"

    # Open the output intent profile
    if {[PDF_load_iccprofile $p "ISOcoated.icc" "usage=outputintent"] == -1} {
	puts stderr [format "Error: %s\n" [PDF_get_errmsg $p]]
        puts stderr "Please install the ICC profile package from \
www.pdflib.com to run the PDF/X-5g starter sample.\n"
	PDF_delete $p
	exit
    }

    # Font embedding is required for PDF/X
    set font [PDF_load_font $p "LuciduxSans-Oblique" "winansi" "embedding"]

    if {$font == -1} {
	puts stderr [format "Error: %s\n" [PDF_get_errmsg $p]]
	PDF_delete $p
	exit
    }

    # Create a template which will serve as proxy. The referenced
    # page (the target) is attached to the proxy.
    # The width and height parameters will be set in PDF_end_template_ext()
    # when we know the size of the target page.
    # The template width and height will be determined automatically,
    # so we don't have to supply them.
    set optlist [format "reference={filename=%s pagenumber=1}" $targetname]
    set proxy [PDF_begin_template_ext $p 0 0 $optlist]

    if {$proxy == -1} {
	puts stderr [format "Error: %s\n" [PDF_get_errmsg $p]]
	PDF_delete $p
	exit
    }

    set width  [PDF_info_image $p $proxy "imagewidth" ""]
    set height [PDF_info_image $p $proxy "imageheight" ""]

    # Draw a crossed-out rectangle to visualize the proxy.
    # Attention: if we use the exact corner points, one half of the
    # linewidth would end up outside the template, and therefore be
    # clipped.
    PDF_setlinewidth $p $linewidth
    PDF_moveto $p [expr $linewidth/2] [expr $linewidth/2]
    PDF_lineto $p [expr $width - $linewidth/2] [expr $linewidth/2]
    PDF_lineto $p [expr $width - $linewidth/2] [expr $height - $linewidth/2]
    PDF_lineto $p [expr $linewidth/2] [expr $height - $linewidth/2]
    PDF_lineto $p [expr $linewidth/2] [expr $linewidth/2]
    PDF_lineto $p [expr $width - $linewidth/2] [expr $height - $linewidth/2]
    
    PDF_moveto $p [expr $width - $linewidth/2] [expr $linewidth/2]
    PDF_lineto $p [expr $linewidth/2] [expr $height - $linewidth/2]
    PDF_stroke $p

    PDF_setfont $p $font 24

    set optlist [format "fitmethod=auto position=center boxsize={%f %f}" $width $height]
    PDF_fit_textline $p "Proxy replaces target here" 0 0 $optlist

    PDF_end_template_ext $p 0 0

    # Create the page
    PDF_begin_page_ext $p 595 842 ""

    PDF_setfont $p $font 18

    PDF_fit_textline $p "PDF/X-5 starter sample with reference to an external page" 50 700 ""

    # Place the proxy on the page
    PDF_fit_image $p $proxy 50 50 "boxsize={500 500} fitmethod=meet"

    PDF_end_page_ext $p ""
    PDF_end_document $p ""

} result] } {
    puts stderr "$result"
}

# delete the PDFlib object
PDF_delete $p
