#!/usr/bin/perl
# $Id: starter_block.pl,v 1.7 2009/09/25 09:32:57 tm Exp $
#
# Block starter:
# Import a PDF page containing blocks and fill text and image
# blocks with some data. For each addressee of the simulated
# mailing a separate page with personalized information is
# generated.
# A real-world application would of course fill the blocks with data
# retrieved from some external data source.
#
# Required software: PPS 8 or above
# Required data: input PDF, image

use pdflib_pl 8.0;
use strict;

# This is where the data files are. Adjust as necessary.
use constant searchpath => "../data";
use constant outfile => "starter_block.pdf";
use constant infile => "block_template.pdf";
use constant imagefile => "new.jpg";

# Names of the person-related blocks contained on the imported page
use constant addressblocks => qw(name street city);

# Data related to various persons used for personalization
use constant persons => (
    ["Mr Maurizio Moroni", "Strada Provinciale 124", "Reggio Emilia"],
    ["Ms Dominique Perrier", "25, rue Lauriston", "Paris"],
    ["Mr Liu Wong", "55 Grizzly Peak Rd.", "Butte"]
);

# Static text simulates database-driven variable contents
use constant intro => "Dear";
use constant goodbye => "Yours sincerely,\nVictor Kraxi";
use constant announcement =>
    "Our <fillcolor=red>BEST PRICE OFFER<fillcolor=black> includes today:" .
    "\n\n" .
    "Long Distance Glider\nWith this paper rocket you can send all your " .
    "messages even when sitting in a hall or in the cinema pretty near " .
    "the back.\n\n" .
    "Giant Wing\nAn unbelievable sailplane! It is amazingly robust and " .
    "can even do aerobatics. But it is best suited to gliding.\n\n" .
    "Cone Head Rocket\nThis paper arrow can be thrown with big swing. " .
    "We launched it from the roof of a hotel. It stayed in the air a " .
    "long time and covered a considerable distance.\n\n" .
    "Super Dart\nThe super dart can fly giant loops with a radius of 4 " .
    "or 5 meters and cover very long distances. Its heavy cone point is " .
    "slightly bowed upwards to get the lift required for loops.\n\n" .
    "Visit us on our Web site at www.kraxi.com!";

my $p = PDF_new();

eval {
    PDF_set_parameter($p, "SearchPath", searchpath);

    # This means we must check return values of load_font() etc.
    PDF_set_parameter($p, "errorpolicy", "return");

    if (PDF_begin_document($p, outfile,
            "destination={type=fitwindow} pagelayout=singlepage") == -1) {
        die("Error: " . PDF_get_errmsg($p));
    }

    PDF_set_info($p, "Creator", "PDFlib starter sample");
    PDF_set_info($p, "Title", "starter_block");

    # Open the Block template which contains PDFlib Blocks
    my $indoc = PDF_open_pdi_document($p, infile, "");
    if ($indoc == -1) {
        die("Error: " . PDF_get_errmsg($p));
    }

    # Open the first page and clone the page size
    my $inpage = PDF_open_pdi_page($p, $indoc, 1, "cloneboxes");
    if ($inpage == -1) {
        die("Error: " . PDF_get_errmsg($p));
    }

    my $image = PDF_load_image($p, "auto", imagefile, "");

    if ($image == -1) {
        die("Error: " . PDF_get_errmsg($p));
    }

    # Based on the imported page generate several pages with the blocks
    # being filled with data related to different persons
    foreach my $person (persons) {
        # Start the output page with a dummy size
        PDF_begin_page_ext($p, 10, 10, "");

        # Place the imported page on the output page, and clone all
        # page boxes which are present in the input page; this will
        # override the dummy size used in begin_page_ext().
        PDF_fit_pdi_page($p, $inpage, 0, 0, "cloneboxes");

        # Option list for text blocks
        my $optlist = "encoding=winansi embedding";

        # Loop over all person-related blocks. Fill the j-th block with the
        # corresponding entry of the persons array.
        my $j = 0;
        foreach my $adressblock (addressblocks) {
            if (PDF_fill_textblock($p, $inpage, $adressblock,
                    $person->[$j], $optlist) == -1) {
                printf("Warning: %s\n", PDF_get_errmsg($p));
            }
            $j += 1;
        }

        # Fill the "intro" block
        my $buf = sprintf "%s %s,", intro, $person->[0];
        if (PDF_fill_textblock($p, $inpage, "intro", $buf, $optlist) == -1) {
            printf("Warning: %s\n", PDF_get_errmsg($p));
        }

        # Fill the "announcement" block
        if (PDF_fill_textblock($p, $inpage, "announcement", announcement,
                $optlist) == -1) {
            printf("Warning: %s\n", PDF_get_errmsg($p));
        }

        # Fill the "goodbye" block
        if (PDF_fill_textblock($p, $inpage, "goodbye", goodbye,
                $optlist) == -1) {
            printf("Warning: %s\n", PDF_get_errmsg($p));
        }

        # Fill the image block
        if (PDF_fill_imageblock($p, $inpage, "icon", $image, "") == -1) {
            printf("Warning: %s\n", PDF_get_errmsg($p));
        }

        PDF_end_page_ext($p, "");
    }

    PDF_close_pdi_page($p, $inpage);
    PDF_close_pdi_document($p, $indoc);
    PDF_close_image($p, $image);

    PDF_end_document($p, "");
};

if ($@) {
    printf("$0: PDFlib Exception occurred:\n");
    printf(" $@\n");
    exit(1);
}

