#!/usr/bin/perl
# $Id: starter_geospatial.pl,v 1.1 2009/09/11 10:32:16 stm Exp $
# Starter for georeferenced PDF:
# Import an image with a map and add geospatial reference information
#
# Sample map and coordinates:
# We use a map from www.openstreetmap.com; the geospatial coordinates of the
# image edges were also provided by that Web site.
# The coordinate system is WGS84 which is also used for GPS.
#
# Required software: PDFlib/PDFlib+PDI/PPS 8
# Required data: image file and associated geospatial reference information

use pdflib_pl 8.0;
use strict;

# This is where the data files are. Adjust as necessary. 
use constant searchpath => "../data";
use constant outfile => "starter_geospatial.pdf";
use constant imagefile => "munich.png";

# WKT for plain latitude/longitude values in WGS84 
use constant georef => 
    "worldsystem={type=geographic wkt={" .
        "GEOGCS[\"WGS 84\"," .
        "  DATUM[\"WGS_1984\", SPHEROID[\"WGS 84\", 6378137,298.257223563]]," .
        "  PRIMEM[\"Greenwich\", 0.0]," .
        "  UNIT[\"Degree\", 0.01745329251994328]]" .
    "}} linearunit=M areaunit=SQM angularunit=degree";

# world coordinates of the image (in degrees) 
use constant worldpoints => (
    48.145, # latitude of top edge 
    11.565, # longitude of left edge 
    11.59,  # longitude of right edge 
    48.13   # latitude of bottom edge 
);

my $p = PDF_new();

eval {
    my $optlist;
    
    PDF_set_parameter($p, "SearchPath", searchpath);

    # This means we must check return values of load_font() etc. 
    PDF_set_parameter($p, "errorpolicy", "return");

    # Start the document 
    if (PDF_begin_document($p, outfile, "compatibility=1.7ext3") == -1) {
        printf("Error: %s\n", PDF_get_errmsg($p));
        PDF_delete($p);
        exit(2);
    }

    PDF_set_info($p, "Creator", "PDFlib starter sample");
    PDF_set_info($p, "Title", "starter_geospatial");

    # Generate georeference option list 
    # Use the four corners as reference points; (0,0)=lower left etc. 
    my $georefoptlist = sprintf
        "georeference={%s mappoints={0 0  1 0  1 1  0 1} ",
        georef;

    $georefoptlist .= "worldpoints={";

    # lower left corner 
    $georefoptlist .= sprintf "%g %g ", (worldpoints)[3], (worldpoints)[1];
    # lower right corner 
    $georefoptlist .= sprintf "%g %g ", (worldpoints)[3], (worldpoints)[2];
    # upper right corner 
    $georefoptlist .= sprintf "%g %g ", (worldpoints)[0], (worldpoints)[2];
    # upper left corner 
    $georefoptlist .= sprintf "%g %g ", (worldpoints)[0], (worldpoints)[1];

    $georefoptlist .=  "} }";

    # Load the image with geospatial reference attached 
    my $image = PDF_load_image($p, "auto", imagefile, $georefoptlist);
    if ($image == -1) {
        printf("Error: %s\n", PDF_get_errmsg($p));
        PDF_delete($p);
        exit(2);
    }

    PDF_begin_page_ext($p, 0, 0, "width=a4.width height=a4.height");

    # Create caption 
    PDF_fit_textline($p, "Map with geospatial reference information",
        50, 700,
        "fontname=LuciduxSans-Oblique encoding=winansi fontsize=18");

    # Place the map on the page 
    PDF_fit_image($p, $image, 50, 50, "boxsize={500 600} fitmethod=meet");

    PDF_end_page_ext($p, "");

    PDF_end_document($p, "");
};

if ($@) {
    printf("$0: PDFlib Exception occurred:\n");
    printf(" $@\n");
    exit(1);
}
