#!/usr/bin/perl
# $Id: starter_pdfx4.pl,v 1.3 2009/11/24 09:22:29 rjs Exp $
#
# PDF/X-4 starter:
# Create PDF/X-4 conforming output with layer variants and transparency
#
# A low-level layer is created for each of several languages, as well
# as an image layer. Each of the language layers together with the
# image layer forms a "layer variant" according to PDF/X-4 (in Acrobat
# layer variants are called "configurations").
# This ensures that low-level layers cannot be enabled/disabled individually,
# but only via the corresponding layer variant. This prevents accidental
# printing of a language layer without the required image layer.
#
# The document contains transparent text which is allowed in
# PDF/X-4, but not earlier PDF/X standards.
#
# Required software: PDFlib/PDFlib+PDI/PPS 8
# Required data: font file, image file, ICC output intent profile
#                (see www.pdflib.com for ICC profiles)

use PDFlib::PDFlib 8.0;
use strict;

# This is where the data files are. Adjust as necessary.
use constant searchpath => "../data";
use constant imagefile => "zebra.tif";

my $p = new PDFlib::PDFlib;

eval {
    # This means we must check return values of load_font() etc.
    $p->set_parameter("errorpolicy", "return");

    $p->set_parameter("SearchPath", searchpath);

    if ($p->begin_document("starter_pdfx4.pdf", "pdfx=PDF/X-4")
            == -1) {
        printf("Error: %s\n", $p->get_errmsg());
        exit(2);
    }

    $p->set_info("Creator", "PDFlib starter sample");
    $p->set_info("Title", "starter_pdfx4");

    if ($p->load_iccprofile("ISOcoated.icc", "usage=outputintent") == -1)
    {
        printf("Error: %s\n", $p->get_errmsg());
        printf("Please install the ICC profile package from " .
               "www.pdflib.com to run the PDF/X-4 starter sample.\n");
        exit(2);
    }

    # Define the low-level layers. These cannot be controlled directly
    # in Acrobat's layer pane.

    my $layer_english = $p->define_layer("English text", "");
    my $layer_german  = $p->define_layer("German text", "");
    my $layer_french  = $p->define_layer("French text", "");
    my $layer_image   = $p->define_layer("Images", "");

    # Define a radio button relationship for the language layers.
    # Individual layers will only be visible in Acrobat X (but
    # not Acrobat 9).
    #
    my $optlist = sprintf
	    "group={%d %d %d}",
	    $layer_english, $layer_german, $layer_french;
    $p->set_layer_dependency("Radiobtn", $optlist);

    # Define the layer combinations for document variants. The variants
    # control the low-level layers, and can be activated in Acrobat 9's
    # layer pane. Using layer variants we can make sure that the image
    # layer cannot accidentally be disabled; it will always accompany
    # the text regardless of the selected language.

    $optlist = sprintf
    	"variantname={English variant} includelayers={%d %d} " .
        	"defaultvariant=true createorderlist",
    	$layer_english, $layer_image;
    $p->set_layer_dependency("Variant", $optlist);

    $optlist = sprintf
    	"variantname={German variant} includelayers={%d %d}",
    	$layer_german, $layer_image;
    $p->set_layer_dependency("Variant", $optlist);

    $optlist = sprintf
    	"variantname={French variant} includelayers={%d %d}",
    	$layer_french, $layer_image;
    $p->set_layer_dependency("Variant", $optlist);

    $p->begin_page_ext(595, 842, "");

    # Font embedding is required for PDF/X
    my $font = $p->load_font("LuciduxSans-Oblique", "winansi", "embedding");

    if ($font == -1) {
        printf("Error: %s\n", $p->get_errmsg());
        exit(2);
    }

    $p->setfont($font, 24);

    $p->begin_layer($layer_english);

    $p->fit_textline("PDF/X-4 starter sample with layers", 50, 700, "");

    $p->begin_layer($layer_german);
    $p->fit_textline("PDF/X-4 Starter-Beispiel mit Ebenen", 50, 700, "");

    $p->begin_layer($layer_french);
    $p->fit_textline("PDF/X-4 Starter exemple avec des calques", 50, 700, "");

    $p->begin_layer($layer_image);

    $p->setfont($font, 48);

    # The RGB image needs an ICC profile; we use sRGB.
    my $icc = $p->load_iccprofile("sRGB", "");
    $optlist = sprintf "iccprofile=%d", $icc;
    my $image = $p->load_image("auto", imagefile, $optlist);

    if ($image == -1) {
        printf("Error: %s\n", $p->get_errmsg());
        exit(2);
    }

    # Place a diagonal stamp across the image area
    my $width = $p->info_image($image, "width", "");
    my $height = $p->info_image($image, "height", "");

    $optlist = sprintf "boxsize={%f %f} stamp=ll2ur", $width, $height;
    $p->fit_textline("Zebra", 0, 0, $optlist);

    # Set transparency in the graphics state
    my $gstate = $p->create_gstate("opacityfill=0.5");
    $p->set_gstate($gstate);

    # Place the image on the page and close it
    $p->fit_image($image, 0.0, 0.0, "");
    $p->close_image($image);

    # Close all layers
    $p->end_layer();

    $p->end_page_ext("");

    $p->end_document("");
};

if ($@) {
    die("$0: PDFlib Exception occurred:\n$@");
}
