#!/usr/bin/perl
# $Id: starter_portfolio.pl,v 1.2 2009/11/23 14:31:30 rjs Exp $
#
# PDF portfolio starter:
# Package multiple PDF and other documents into a PDF portfolio
# The generated PDF portfolio requires Acrobat 9 for proper
# viewing. The documents in the Portfolio will be assigned predefined
# and custom metadata fields; for the custom fields a schema description
# is created.
#
# Acrobat 8 will only display a "PDF package" with a flat list of documents
# without any folder structure.
#
# Required software: PDFlib/PDFlib+PDI/PPS 8
# Required data: PDF and other input documents

use PDFlib::PDFlib 8.0;
use strict;

# This is where the data files are. Adjust as necessary.
use constant searchpath => "../data";

# Key names used to make a dictionary for the description of the
# portfolio entries
my @document_keys = qw(filename description status id);

# Function to create a hash describing each document
sub make_document_hash {
    my $values = $_[0]; # reference to array
    my %result;
    @result{@document_keys} = @{$values};
    return \%result;
}

# The documents for the Portfolio along with description and metadata,
# organized as an array of references to hashes
my @root_folder_docs = map { make_document_hash($_) } (
    [
        "TIR_____.AFM",             # filename
        "Metrics for Times-Roman",  # description
        "internal",                 # status
        200                         # id
    ],
    [
        "nesrin.jpg",
        "Zabrisky point",
        "archived",
        300
    ]
);

my @datasheet_docs = map { make_document_hash($_) } (
    [
        "PDFlib-real-world.pdf",
        "PDFlib in the real world",
        "published",
        100
    ],
    [
        "PDFlib-datasheet.pdf",
        "Generate PDF on the fly",
        "published",
        101
    ],
    [
        "TET-datasheet.pdf",
        "Extract text and images from PDF",
        "published",
        102
    ],
    [
        "PLOP-datasheet.pdf",
        "PDF Linearization, Optimization, Protection",
        "published",
        103
    ],
    [
        "pCOS-datasheet.pdf",
        "PDF Information Retrieval Tool",
        "published",
        104
    ]
);

my $p = new PDFlib::PDFlib;

eval {
    my ($optlist, $doc);
    
    $p->set_parameter("SearchPath", searchpath);

    # This means we must check return values of load_font() etc.
    $p->set_parameter("errorpolicy", "return");

    if ($p->begin_document("starter_portfolio.pdf", "compatibility=1.7ext3") == -1) {
        printf("Error: %s\n", $p->get_errmsg());
        exit(2);
    }

    $p->set_info("Creator", "PDFlib starter sample");
    $p->set_info("Title", "starter_portfolio");

    # Insert all files for the root folder along with their description
    # and the following custom fields:
    # status   string describing the document status
    # id       numerical identifier, prefixed with "PHX"
    foreach $doc (@root_folder_docs) {
        $optlist = sprintf
            "description={%s} " .
            "fieldlist={ " .
                    "{key=status value=%s} " .
                    "{key=id value=%d prefix=PHX type=text} " .
            "}",
            $doc->{description}, $doc->{status}, $doc->{id};

        # -1 means root folder
        $p->add_portfolio_file(-1, $doc->{filename}, $optlist);
    }

    # Create the "datasheets" folder in the root folder
    my $folder = $p->add_portfolio_folder(-1, "datasheets", "");

    # Insert documents in the "datasheets" folder along with
    # description and custom fields
    foreach $doc (@datasheet_docs) {
        $optlist = sprintf
        	"description={%s} " .
        	"fieldlist={ " .
        		"{key=status value=%s} " .
        		"{key=id value=%d prefix=PHX type=text} " .
        	"}",
        	$doc->{description}, $doc->{status}, $doc->{id};
    
        # Add the file to the "datasheets" folder
        $p->add_portfolio_file($folder, $doc->{filename}, $optlist);
    }
    
    # Create a single-page document as cover sheet
    $p->begin_page_ext(0, 0, "width=a4.width height=a4.height");
    
    my $font = $p->load_font("Helvetica", "winansi", "");
    if ($font == -1) {
        printf("Error: %s\n", $p->get_errmsg());
        exit(2);
    }

    $p->setfont($font, 24);
    $p->fit_textline("Welcome to the PDFlib Portfolio sample!", 50, 700, "");

    $p->end_page_ext("");
    
    # Set options for Portfolio display
    $optlist = "portfolio={initialview=detail ";
    
    # Add schema definition for Portfolio metadata
    $optlist .=
        "schema={ " .
        # Some predefined fields are included here to make them visible.
        "{order=1 label=Name key=_filename visible editable} " .
        "{order=2 label=Description key=_description visible} " .
        "{order=3 label=Size key=_size visible} " .
        "{order=4 label={Last edited} key=_moddate visible} " .
        
        # User-defined fields
        "{order=5 label=Status key=status type=text editable} " .
        "{order=6 label=ID key=id type=text editable} ";
    
    $optlist .= "}}";

    $p->end_document($optlist);
};

if ($@) {
    die("$0: PDFlib Exception occurred:\n$@");
}
