'
' Starter sample for OpenType font features
'
' Demonstrate various typographic OpenType features after checking
' whether a particular feature is supported in a font.
'
' Required data: suitable font with OpenType feature tables
'
' This sample uses a default font which includes various features.
' Depending on the implementation of the features you
' may have to replace the sample text below.
'

Imports System
Imports System.Text
Imports PDFlib_dotnet

Structure testcase
    Public description As String
    ' the default test font above will be used if this string is empty
    Public fontname As String
    Public feature As String
    Public text As String
    ' Constructor:
    Public Sub New(ByVal s_description As String, ByVal s_fontname As String, _
            ByVal s_feature As String, ByVal s_text As String)
        description = s_description
        fontname = s_fontname
        feature = s_feature
        text = s_text
    End Sub
End Structure

Module starter_opentype
  Sub Main()
    ' This is where the data files are. Adjust as necessary.
    Dim searchpath As String = "../../../data"
    Dim outfile As String = "starter_opentype.pdf"

    Dim optlist As New StringBuilder
    Dim p As PDFlib = Nothing
    Dim row, col, table, test, font As Integer
    Dim llx As Double = 50
    Dim lly As Double = 50
    Dim urx As Double = 800
    Dim ury As Double = 550
    Dim result As String

    ' This font will be used unless another one is specified in the table
    Dim defaulttestfont As String = "NotoSerif-Regular"

    Dim header() As String = { _
        "OpenType feature", _
        "Option list", _
        "Font name", _
        "Raw input (feature disabled)", _
        "Feature enabled" _
    }
    Dim MAXCOL As Integer = header.Length

        Dim testcases() As testcase = { _
            New testcase( _
              "ligatures", _
              "", _
              "liga", _
               "ff fi fl ffi ffl" _
            ), _
            New testcase( _
              "discretionary ligatures", _
              "", _
              "dlig", _
               "ch tz" _
            ), _
            New testcase( _
              "small capitals", _
              "", _
              "smcp", _
               "PostScript" _
            ), _
            New testcase( _
              "ordinals", _
              "", _
              "ordn", _
               "1o 2a 3o" _
            ), _
            New testcase( _
              "fractions", _
              "", _
              "frac", _
               "1/2 1/4 3/4" _
            ), _
            New testcase( _
              "slashed zero", _
              "", _
              "zero", _
               "0" _
            ), _
            New testcase( _
              "historical forms", _
              "", _
              "hist", _
               "s" _
            ), _
            New testcase( _
              "proportional figures", _
              "", _
              "pnum", _
               "0123456789" _
            ), _
            New testcase( _
              "old-style figures", _
              "", _
              "onum", _
               "0123456789" _
            ), _
            New testcase( _
              "lining figures", _
              "", _
              "lnum", _
               "0123456789" _
            ), _
            New testcase( _
              "superscript", _
              "", _
              "sups", _
               "123" _
            ), _
            New testcase( _
              "subscript", _
              "", _
              "subs", _
               "H2SO4" _
            ) _
        }
    Dim n_testcases As Integer = testcases.Length

    Try
        ' create a New PDFlib object
        p = New PDFlib()

	' Set the search path for fonts and PDF files 
	p.set_option("SearchPath={{"+searchpath + "}} charref=true")

        ' This means that formatting and other errors will raise an
        ' exception. This simplifies our sample code, but is not
        ' recommended for production code.
        '/
        p.set_option("errorpolicy=exception")

        ' Set an output path according to the name of the topic
        if (p.begin_document(outfile, "") = -1) Then
            Console.WriteLine("Error: {0}\n", p.get_errmsg())
            return
        End If

        p.set_info("Creator", "PDFlib starter sample")
        p.set_info("Title", "starter_opentype")

        table = -1

        ' Table header
        row = 1
        For col=1 To MAXCOL Step 1
            optlist.Length = 0
            optlist.Append("fittextline={fontname=NotoSerif-Regular ")
            optlist.Append("fontsize=12} margin=4")
            table = p.add_table_cell(table, col, row, header(col-1), _
                optlist.ToString())
        NEXT

        ' Create a table with feature samples, one feature per table row
        row = 2
        For test=0 To n_testcases - 1 Step 1
            Dim testfont As String
            Dim buf As New StringBuilder

            ' Use the entry in the test table if available, and the
            ' default test font otherwise. This way we can easily check
            ' a font for all features, as well as insert suitable fonts
            ' for individual features.
            '/
            If (testcases(test).fontname <> "") Then
                testfont = testcases(test).fontname
            Else
                testfont = defaulttestfont
            End If

            col=1

            ' Common option list for columns 1-3
            optlist.Length = 0
            optlist.Append("fittextline={fontname=NotoSerif-Regular ")
            optlist.Append("fontsize=12} margin=4")

            ' Column 1: feature description
            table = p.add_table_cell(table, col, row, _
                testcases(test).description, optlist.ToString())
            col = col + 1

            ' Column 2: option list
            buf.Length = 0
            buf.Append("features={")
            buf.AppendFormat("{0}", testcases(test).feature)
            buf.Append("}")
            table = p.add_table_cell(table, col, row, buf.ToString(), optlist.ToString())
            col = col + 1

            ' Column 3: font name
            table = p.add_table_cell(table, col, row, testfont, optlist.ToString())
            col = col + 1

            ' Column 4: raw input text with  feature disabled
            optlist.Length = 0
            optlist.Append("fittextline={fontname={")
            optlist.AppendFormat("{0}", testfont)
            optlist.Append("} fontsize=12} margin=4")
            table = p.add_table_cell(table, col, row, _
                    testcases(test).text, optlist.ToString())
            col = col + 1

            ' Column 5: text with enabled feature, or warning if the
            ' feature is not available in the font

            font = p.load_font(testfont, "unicode", "")

            ' Check whether font contains the required feature table
            optlist.Length = 0
            optlist.AppendFormat( "name={0}", testcases(test).feature)
            If (p.info_font(font, "feature", optlist.ToString()) = 1) Then
                ' feature is available: apply it to the text
                optlist.Length = 0
                optlist.Append("fittextline={fontname={")
                optlist.AppendFormat("{0}", testfont)
                optlist.Append("} fontsize=12 features={")
                optlist.AppendFormat("{0}", testcases(test).feature)
                optlist.Append("}} margin=4")
                table = p.add_table_cell(table, col, row, _
                   testcases(test).text, optlist.ToString())
                col = col + 1
            Else
                ' feature is not available: emit a warning
                optlist.Length = 0
                optlist.Append( _
                     "fittextline={fontname=NotoSerif-Regular " & _
                     "fontsize=12 fillcolor=red} margin=4")
                table = p.add_table_cell(table, col, row, _
                    "(feature not available in this font)", optlist.ToString())
                col = col + 1
            End If

            row = row + 1
        Next

        ' Loop until all of the table is placed; create new pages
        ' as long as more table instances need to be placed.
        '/
        do
            p.begin_page_ext(0, 0, "width=a4.height height=a4.width")

            ' Shade every other row; draw lines for all table cells.
            optlist.Length = 0
            optlist.Append( "header=1 fill={{area=rowodd " & _
                "fillcolor={gray 0.9}}} stroke={{line=other}} ")

            ' Place the table instance
            result = p.fit_table(table, llx, lly, urx, ury, optlist.ToString())

            if (result = "_error") Then
                Console.WriteLine("Couldn't place table: {0}\n", p.get_errmsg())
                return
            End If

            p.end_page_ext("")

        Loop While (result = "_boxfull")

        p.end_document("")

    Catch e As PDFlibException
        Console.Error.WriteLine("PDFlib exception occurred:")
        Console.Error.WriteLine("[{0}] {1}: {2}", e.get_errnum(), e.get_apiname(), e.get_errmsg)
    Catch e As System.Exception
        Console.Error.WriteLine(e.ToString())
    Finally
        If Not p Is Nothing Then
            p.Dispose()
            p = Nothing
        End If
    End Try
  End Sub
End Module
