//
// PDF/X-4 starter:
// Create PDF/X-4 conforming output with layers and transparency
//
// A low-level layer is created for each of several languages, as well
// as an image layer. 
//
// The document contains transparent text which is allowed in
// PDF/X-4, but not earlier PDF/X standards.
//
// Required data: font file, image file, ICC output intent profile
//                (see www.pdflib.com for output intent ICC profiles)
//

using System;
using System.Text;
using PDFlib_dotnet;

class starter_pdfx4 {
  static int Main(string[] args)
  {
    /* This is where the data files are. Adjust as necessary.*/
    string searchpath = "../../../data";

    PDFlib p = null;
    string imagefile = "zebra.tif";
    StringBuilder optlist = new StringBuilder();

    int font, image;
    int layer_english, layer_german, layer_french, layer_image;
    double width, height;

    try {
        /* create a new PDFlib object */
        p = new PDFlib();

        // This means we must check return values of load_font() etc.
        p.set_option("errorpolicy=return");

        // Set the search path for fonts and PDF files 
        p.set_option("SearchPath={{" + searchpath + "}}");


        if (p.begin_document("starter_pdfx4.pdf", "pdfx=PDF/X-4") == -1) {
            Console.WriteLine("Error: {0}\n", p.get_errmsg());
            return(2);
        }

        p.set_info("Creator", "PDFlib starter sample");
        p.set_info("Title", "starter_pdfx4");


        if (p.load_iccprofile("ISOcoated_v2_eci.icc", "usage=outputintent") == -1)
        {
            Console.WriteLine("Error: {0}\n", p.get_errmsg());
            Console.WriteLine("See www.pdflib.com for output intent ICC profiles.\n");
            return(2);
        }

        /* Define the layers; "defaultstate" specifies whether or not
         * the layer is visible when the page is opened.
         */

        layer_english = p.define_layer("English text", "defaultstate=true");
        layer_german  = p.define_layer("German text", "defaultstate=false");
        layer_french  = p.define_layer("French text", "defaultstate=false");

        /* Define a radio button relationship for the language layers.  */
        optlist.Length = 0;
        optlist.AppendFormat("group={{ {0} {1} {2} }}",
        	layer_english, layer_german, layer_french);
        p.set_layer_dependency("Radiobtn", optlist.ToString());

        layer_image   = p.define_layer("Images", "defaultstate=true");

        p.begin_page_ext(0,0, "width=a4.width height=a4.height");

        font = p.load_font("NotoSerif-Regular", "winansi", "");

        if (font == -1) {
            Console.WriteLine("Error: {0}\n", p.get_errmsg());
            return(2);
        }

        p.setfont(font, 24);

        p.begin_layer(layer_english);

        p.fit_textline("PDF/X-4 starter sample with layers", 50, 700, "");

        p.begin_layer(layer_german);
        p.fit_textline("PDF/X-4 Starter-Beispiel mit Ebenen", 50, 700, "");

        p.begin_layer(layer_french);
        p.fit_textline("PDF/X-4 Starter exemple avec des calques", 50, 700, "");

        p.begin_layer(layer_image);

        image = p.load_image("auto", imagefile,"");
        if (image == -1) {
            return(2);
        }

        /* Place the image on the page */
        p.fit_image(image, (double) 0.0, (double) 0.0, "");

        /* Place a diagonal stamp across the image area */
        width = p.info_image(image, "width", "");
        height = p.info_image(image, "height", "");

        optlist.Length = 0;
        optlist.Append("boxsize={");
        optlist.AppendFormat("{0} {1}", width, height);
        optlist.Append("} stamp=ll2ur ");
        optlist.AppendFormat("font={0} ", font);
        optlist.Append("fillcolor={lab 100 0 0} gstate={opacityfill=0.5}");
        p.fit_textline("Zebra", 0, 0, optlist.ToString());

        p.close_image(image);

        /* Close all layers */
        p.end_layer();

        p.end_page_ext("");

        p.end_document("");
    }

    catch (PDFlibException e)
    {
        // caught exception thrown by PDFlib
        Console.WriteLine("PDFlib exception occurred:");
        Console.WriteLine("[{0}] {1}: {2}\n", e.get_errnum(),
                e.get_apiname(), e.get_errmsg());
    } finally {
        if (p != null) {
            p.Dispose();
        }
    }
    return(0);
  }
}
