/*---------------------------------------------------------------------------*
 |          Copyright (c) 1997-2022 PDFlib GmbH. All rights reserved.        |
 +---------------------------------------------------------------------------+
 |    This software may not be copied or distributed except as expressly     |
 |    authorized by PDFlib GmbH's general license agreement or a custom      |
 |    license agreement signed by PDFlib GmbH.                               |
 |    For more information about licensing please refer to www.pdflib.com.   |
 *---------------------------------------------------------------------------*/

/* $Id$
 *
 * PDFlib Java class
 */

package com.pdflib;

/** PDFlib -- a library for generating PDF on the fly.

    Note that this is only a syntax summary. It covers several products:
    PDFlib, PDFlib+PDI, and PDFlib Personalization Server (PPS).
    Some methods will work only if a suitable license key for the product
    class has been provided. A comparison which details function availability
    in different products is available at http://www.pdflib.com.

    For complete information please refer to the PDFlib API reference
    manual which is available as PDF file in the "doc" directory of the
    PDFlib distribution.

    @author PDFlib GmbH
    @version 11.0.0
*/

public final class pdflib implements IpCOS {

    // The initialization code for loading the PDFlib shared library.
    // The library name will be transformed into something platform-
    // specific by the VM, e.g. libpdflib_java.so or pdflib_java.dll.

    static {
	try {
	    System.loadLibrary("pdflib_java");
	} catch (UnsatisfiedLinkError e) {
	    System.err.println(
	"Cannot load the PDFlib shared library/DLL for Java.\n" +
	"Make sure to properly install the native PDFlib library.\n\n" +
	"For your information, the current value of java.library.path is:\n" +
	 System.getProperty("java.library.path") + "\n");

	    throw e;
	}
    }

    // ------------------------------------------------------------------------
    // public functions
    /** Activate a previously created structure element or other content item.
        @param id id
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void activate_item(int id)
    throws PDFlibException
    {
	PDF_activate_item(p, id);
    }

    /** Create a named destination on a page in the document.
        @param name name
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void add_nameddest(String name, String optlist)
    throws PDFlibException
    {
	PDF_add_nameddest(p, name, optlist);
    }

    /** Add a point to a new or existing path object.
        @return  A path handle which can be used in subsequent path-related calls.
        @param path path
        @param x x
        @param y y
        @param type type
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int add_path_point(int path, double x, double y, String type, String optlist)
    throws PDFlibException
    {
	return PDF_add_path_point(p, path, x, y, type, optlist);
    }

    /** Add a file to a portfolio folder or a package.
        @return  -1 on error, and 1 otherwise.
        @param folder folder
        @param filename filename
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int add_portfolio_file(int folder, String filename, String optlist)
    throws PDFlibException
    {
	return PDF_add_portfolio_file(p, folder, filename, optlist);
    }

    /** Add a folder to a new or existing portfolio.
        @return  A folder handle which can be used in subsequent portfolio-related calls.
        @param parent parent
        @param foldername foldername
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int add_portfolio_folder(int parent, String foldername, String optlist)
    throws PDFlibException
    {
	return PDF_add_portfolio_folder(p, parent, foldername, optlist);
    }

    /** Add a cell to a new or existing table.
        @return  A table handle which can be used in subsequent table-related calls.
        @param table table
        @param column column
        @param row row
        @param text text
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int add_table_cell(int table, int column, int row, String text, String optlist)
    throws PDFlibException
    {
	return PDF_add_table_cell(p, table, column, row, text, optlist);
    }

    /** Create a Textflow object, or add text and explicit options to an existing Textflow.
        @return  A Textflow handle, or -1 on error.
        @param textflow textflow
        @param text text
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int add_textflow(int textflow, String text, String optlist)
    throws PDFlibException
    {
	return PDF_add_textflow(p, textflow, text, optlist);
    }

    /** Align the coordinate system with a relative vector.
        @param dx dx
        @param dy dy
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void align(double dx, double dy)
    throws PDFlibException
    {
	PDF_align(p, dx, dy);
    }

    /** Draw a counterclockwise circular arc segment.
        @param x x
        @param y y
        @param r r
        @param alpha alpha
        @param beta beta
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void arc(double x, double y, double r, double alpha, double beta)
    throws PDFlibException
    {
	PDF_arc(p, x, y, r, alpha, beta);
    }

    /** Draw a clockwise circular arc segment.
        @param x x
        @param y y
        @param r r
        @param alpha alpha
        @param beta beta
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void arcn(double x, double y, double r, double alpha, double beta)
    throws PDFlibException
    {
	PDF_arcn(p, x, y, r, alpha, beta);
    }

    /** Create a new PDF file subject to various options.
        @return  -1 on error, and 1 otherwise.
        @param filename filename
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int begin_document(String filename, String optlist)
    throws PDFlibException
    {
	return PDF_begin_document(p, filename, optlist);
    }

    /** Create a new node in the document part hierarchy (requires PDF/VT or   PDF 2.0).
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void begin_dpart(String optlist)
    throws PDFlibException
    {
	PDF_begin_dpart(p, optlist);
    }

    /** Start a Type 3 font definition.
        @param fontname fontname
        @param a a
        @param b b
        @param c c
        @param d d
        @param e e
        @param f f
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void begin_font(String fontname, double a, double b, double c, double d, double e, double f, String optlist)
    throws PDFlibException
    {
	PDF_begin_font(p, fontname, a, b, c, d, e, f, optlist);
    }

    /** Start a glyph definition for a Type 3 font.
        @param uv uv
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void begin_glyph_ext(int uv, String optlist)
    throws PDFlibException
    {
	PDF_begin_glyph_ext(p, uv, optlist);
    }

    /** Open a structure element or other content item with attributes supplied as options.
        @return  An item handle.
        @param tagname tagname
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int begin_item(String tagname, String optlist)
    throws PDFlibException
    {
	return PDF_begin_item(p, tagname, optlist);
    }

    /** Start a layer for subsequent output on the page.
        @param layer layer
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void begin_layer(int layer)
    throws PDFlibException
    {
	PDF_begin_layer(p, layer);
    }

    /** Begin a marked content sequence with optional properties.
        @param tagname tagname
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void begin_mc(String tagname, String optlist)
    throws PDFlibException
    {
	PDF_begin_mc(p, tagname, optlist);
    }

    /** Add a new page to the document, and specify various options.
        @param width width
        @param height height
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void begin_page_ext(double width, double height, String optlist)
    throws PDFlibException
    {
	PDF_begin_page_ext(p, width, height, optlist);
    }

    /** Start a pattern definition with options.
        @return  A pattern handle.
        @param width width
        @param height height
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int begin_pattern_ext(double width, double height, String optlist)
    throws PDFlibException
    {
	return PDF_begin_pattern_ext(p, width, height, optlist);
    }

    /** Start a template definition.
        @return  A template handle.
        @param width width
        @param height height
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int begin_template_ext(double width, double height, String optlist)
    throws PDFlibException
    {
	return PDF_begin_template_ext(p, width, height, optlist);
    }

    /** Draw a circle.
        @param x x
        @param y y
        @param r r
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void circle(double x, double y, double r)
    throws PDFlibException
    {
	PDF_circle(p, x, y, r);
    }

    /** Draw a circular arc segment defined by three points.
        @param x1 x1
        @param y1 y1
        @param x2 x2
        @param y2 y2
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void circular_arc(double x1, double y1, double x2, double y2)
    throws PDFlibException
    {
	PDF_circular_arc(p, x1, y1, x2, y2);
    }

    /** Use the current path as clipping path, and terminate the path.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void clip()
    throws PDFlibException
    {
	PDF_clip(p);
    }

    /** Close an open font handle which has not yet been used in the document.
        @param font font
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void close_font(int font)
    throws PDFlibException
    {
	PDF_close_font(p, font);
    }

    /** Close vector graphics.
        @param graphics graphics
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void close_graphics(int graphics)
    throws PDFlibException
    {
	PDF_close_graphics(p, graphics);
    }

    /** Close an image or template.
        @param image image
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void close_image(int image)
    throws PDFlibException
    {
	PDF_close_image(p, image);
    }

    /** Close all open PDI page handles, and close the input PDF document.
        @param doc doc
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void close_pdi_document(int doc)
    throws PDFlibException
    {
	PDF_close_pdi_document(p, doc);
    }

    /** Close the page handle and free all page-related resources.
        @param page page
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void close_pdi_page(int page)
    throws PDFlibException
    {
	PDF_close_pdi_page(p, page);
    }

    /** Close the current path.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void closepath()
    throws PDFlibException
    {
	PDF_closepath(p);
    }

    /** Close the path, fill, and stroke it.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void closepath_fill_stroke()
    throws PDFlibException
    {
	PDF_closepath_fill_stroke(p);
    }

    /** Close the path, and stroke it.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void closepath_stroke()
    throws PDFlibException
    {
	PDF_closepath_stroke(p);
    }

    /** Apply a transformation matrix to the current coordinate system.
        @param a a
        @param b b
        @param c c
        @param d d
        @param e e
        @param f f
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void concat(double a, double b, double c, double d, double e, double f)
    throws PDFlibException
    {
	PDF_concat(p, a, b, c, d, e, f);
    }

    /** Print text at the next line.
        @param text text
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void continue_text(String text)
    throws PDFlibException
    {
	PDF_continue_text(p, text);
    }

    /** Convert a string in an arbitrary encoding to a Unicode string in various formats.
        @return  The converted Unicode string.
        @param inputformat inputformat
        @param inputstring inputstring
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final String convert_to_unicode(String inputformat, byte[] inputstring, String optlist)
    throws PDFlibException
    {
	return PDF_convert_to_unicode(p, inputformat, inputstring, optlist);
    }

    /** Create a 3D view.
        @return  A 3D view handle, or -1 on error.
        @param username username
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int create_3dview(String username, String optlist)
    throws PDFlibException
    {
	return PDF_create_3dview(p, username, optlist);
    }

    /** Create an action which can be applied to various objects and events.
        @return  An action handle.
        @param type type
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int create_action(String type, String optlist)
    throws PDFlibException
    {
	return PDF_create_action(p, type, optlist);
    }

    /** Create an annotation on the current page.
        @param llx llx
        @param lly lly
        @param urx urx
        @param ury ury
        @param type type
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void create_annotation(double llx, double lly, double urx, double ury, String type, String optlist)
    throws PDFlibException
    {
	PDF_create_annotation(p, llx, lly, urx, ury, type, optlist);
    }

    /** Create a DeviceN colorspace with an arbitrary number of color components.
        @return  A DeviceN color space handle, or -1 on error.
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int create_devicen(String optlist)
    throws PDFlibException
    {
	return PDF_create_devicen(p, optlist);
    }

    /** Create a bookmark subject to various options.
        @return  A handle for the generated bookmark.
        @param text text
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int create_bookmark(String text, String optlist)
    throws PDFlibException
    {
	return PDF_create_bookmark(p, text, optlist);
    }

    /** Create a new form field or fill an imported form field.
        @param llx llx
        @param lly lly
        @param urx urx
        @param ury ury
        @param name name
        @param type type
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void create_field(double llx, double lly, double urx, double ury, String name, String type, String optlist)
    throws PDFlibException
    {
	PDF_create_field(p, llx, lly, urx, ury, name, type, optlist);
    }

    /** Create a form field group subject to various options.
        @param name name
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void create_fieldgroup(String name, String optlist)
    throws PDFlibException
    {
	PDF_create_fieldgroup(p, name, optlist);
    }

    /** Create a graphics state object subject to various options.
        @return  A graphic state handle.
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int create_gstate(String optlist)
    throws PDFlibException
    {
	return PDF_create_gstate(p, optlist);
    }

    /** Create a named virtual read-only file from data provided in memory.
        @param filename filename
        @param data data
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void create_pvf(String filename, byte[] data, String optlist)
    throws PDFlibException
    {
	PDF_create_pvf(p, filename, data, optlist);
    }

    /** Create a Textflow object from text contents, inline options, and explicit options.
        @return  A Textflow handle, or -1 on error.
        @param text text
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int create_textflow(String text, String optlist)
    throws PDFlibException
    {
	return PDF_create_textflow(p, text, optlist);
    }

    /** Draw a Bezier curve from the current point, using 3 more control points.
        @param x1 x1
        @param y1 y1
        @param x2 x2
        @param y2 y2
        @param x3 x3
        @param y3 y3
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void curveto(double x1, double y1, double x2, double y2, double x3, double y3)
    throws PDFlibException
    {
	PDF_curveto(p, x1, y1, x2, y2, x3, y3);
    }

    /** Create a new layer definition.
        @return  A layer handle which can be used in subsequent layer-related calls.
        @param name name
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int define_layer(String name, String optlist)
    throws PDFlibException
    {
	return PDF_define_layer(p, name, optlist);
    }

    /** Delete a path object.
        @param path path
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void delete_path(int path)
    throws PDFlibException
    {
	PDF_delete_path(p, path);
    }

    /** Delete a named virtual file and free its data structures (but not the contents).
        @return  -1 if the virtual file exists but is locked, and 1 otherwise.
        @param filename filename
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int delete_pvf(String filename)
    throws PDFlibException
    {
	return PDF_delete_pvf(p, filename);
    }

    /** Delete a table and all associated data structures.
        @param table table
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void delete_table(int table, String optlist)
    throws PDFlibException
    {
	PDF_delete_table(p, table, optlist);
    }

    /** Delete a textflow and all associated data structures.
        @param textflow textflow
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void delete_textflow(int textflow)
    throws PDFlibException
    {
	PDF_delete_textflow(p, textflow);
    }

    /** Download data from a network resource and store it in a disk-based or virtual file.
        @return  -1 on error, and 1 otherwise.
        @param filename filename
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int download(String filename, String optlist)
    throws PDFlibException
    {
	return PDF_download(p, filename, optlist);
    }

    /** Draw a path object.
        @param path path
        @param x x
        @param y y
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void draw_path(int path, double x, double y, String optlist)
    throws PDFlibException
    {
	PDF_draw_path(p, path, x, y, optlist);
    }

    /** Draw an ellipse.
        @param x x
        @param y y
        @param rx rx
        @param ry ry
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void ellipse(double x, double y, double rx, double ry)
    throws PDFlibException
    {
	PDF_ellipse(p, x, y, rx, ry);
    }

    /** Draw an elliptical arc segment from the current point.
        @param x x
        @param y y
        @param rx rx
        @param ry ry
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void elliptical_arc(double x, double y, double rx, double ry, String optlist)
    throws PDFlibException
    {
	PDF_elliptical_arc(p, x, y, rx, ry, optlist);
    }

    /** Add a glyph name and/or Unicode value to a custom 8-bit encoding.
        @param encoding encoding
        @param slot slot
        @param glyphname glyphname
        @param uv uv
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void encoding_set_char(String encoding, int slot, String glyphname, int uv)
    throws PDFlibException
    {
	PDF_encoding_set_char(p, encoding, slot, glyphname, uv);
    }

    /** Close the generated PDF document and apply various options.
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void end_document(String optlist)
    throws PDFlibException
    {
	PDF_end_document(p, optlist);
    }

    /** Close a node in the document part hierarchy (requires PDF/VT or PDF 2.0).
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void end_dpart(String optlist)
    throws PDFlibException
    {
	PDF_end_dpart(p, optlist);
    }

    /** Terminate a Type 3 font definition.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void end_font()
    throws PDFlibException
    {
	PDF_end_font(p);
    }

    /** Terminate a glyph definition for a Type 3 font.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void end_glyph()
    throws PDFlibException
    {
	PDF_end_glyph(p);
    }

    /** Close a structure element or other content item.
        @param id id
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void end_item(int id)
    throws PDFlibException
    {
	PDF_end_item(p, id);
    }

    /** Deactivate all active layers.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void end_layer()
    throws PDFlibException
    {
	PDF_end_layer(p);
    }

    /** End the least recently opened marked content sequence.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void end_mc()
    throws PDFlibException
    {
	PDF_end_mc(p);
    }

    /** Finish a page, and apply various options.
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void end_page_ext(String optlist)
    throws PDFlibException
    {
	PDF_end_page_ext(p, optlist);
    }

    /** Finish a pattern definition.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void end_pattern()
    throws PDFlibException
    {
	PDF_end_pattern(p);
    }

    /** Finish a template definition.
        @param width width
        @param height height
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void end_template_ext(double width, double height)
    throws PDFlibException
    {
	PDF_end_template_ext(p, width, height);
    }

    /** End the current path without filling or stroking it.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void endpath()
    throws PDFlibException
    {
	PDF_endpath(p);
    }

    /** Fill the interior of the path with the current fill color.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void fill()
    throws PDFlibException
    {
	PDF_fill(p);
    }

    /** Fill a graphics block with variable data according to its properties.
        @return  -1 on error, and 1 otherwise.
        @param page page
        @param blockname blockname
        @param graphics graphics
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int fill_graphicsblock(int page, String blockname, int graphics, String optlist)
    throws PDFlibException
    {
	return PDF_fill_graphicsblock(p, page, blockname, graphics, optlist);
    }

    /** Fill an image block with variable data according to its properties.
        @return  -1 on error, and 1 otherwise.
        @param page page
        @param blockname blockname
        @param image image
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int fill_imageblock(int page, String blockname, int image, String optlist)
    throws PDFlibException
    {
	return PDF_fill_imageblock(p, page, blockname, image, optlist);
    }

    /** Fill a PDF block with variable data according to its properties.
        @return  -1 on error, and 1 otherwise.
        @param page page
        @param blockname blockname
        @param contents contents
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int fill_pdfblock(int page, String blockname, int contents, String optlist)
    throws PDFlibException
    {
	return PDF_fill_pdfblock(p, page, blockname, contents, optlist);
    }

    /** Fill a Textline or Textflow Block with variable data according to its properties.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void fill_stroke()
    throws PDFlibException
    {
	PDF_fill_stroke(p);
    }

    /** Fill a Textline or Textflow Block with variable data according to its properties.
        @return  -1 on error, and 1 otherwise.
        @param page page
        @param blockname blockname
        @param text text
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int fill_textblock(int page, String blockname, String text, String optlist)
    throws PDFlibException
    {
	return PDF_fill_textblock(p, page, blockname, text, optlist);
    }

    /** Place vector graphics on a content stream, subject to various options.
        @param graphics graphics
        @param x x
        @param y y
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void fit_graphics(int graphics, double x, double y, String optlist)
    throws PDFlibException
    {
	PDF_fit_graphics(p, graphics, x, y, optlist);
    }

    /** Place an image or template on the page, subject to various options.
        @param image image
        @param x x
        @param y y
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void fit_image(int image, double x, double y, String optlist)
    throws PDFlibException
    {
	PDF_fit_image(p, image, x, y, optlist);
    }

    /** Place an imported PDF page on the page subject to various options.
        @param page page
        @param x x
        @param y y
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void fit_pdi_page(int page, double x, double y, String optlist)
    throws PDFlibException
    {
	PDF_fit_pdi_page(p, page, x, y, optlist);
    }

    /** Fully or partially place a table on the page.
        @return  A string which specifies the reason for returning.
        @param table table
        @param llx llx
        @param lly lly
        @param urx urx
        @param ury ury
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final String fit_table(int table, double llx, double lly, double urx, double ury, String optlist)
    throws PDFlibException
    {
	return PDF_fit_table(p, table, llx, lly, urx, ury, optlist);
    }

    /** Format the next portion of a Textflow.
        @return  A string which specifies the reason for returning.
        @param textflow textflow
        @param llx llx
        @param lly lly
        @param urx urx
        @param ury ury
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final String fit_textflow(int textflow, double llx, double lly, double urx, double ury, String optlist)
    throws PDFlibException
    {
	return PDF_fit_textflow(p, textflow, llx, lly, urx, ury, optlist);
    }

    /** Place a single line of text at position (x, y) subject to various options.
        @param text text
        @param x x
        @param y y
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void fit_textline(String text, double x, double y, String optlist)
    throws PDFlibException
    {
	PDF_fit_textline(p, text, x, y, optlist);
    }

    /** Get the name of the API method which threw the last exception or failed.
        @return  Name of an API method.
    */
    public final String get_apiname()
    {
	return PDF_get_apiname(p);
    }

    /** Get the contents of the PDF output buffer.
        @return  A buffer full of binary PDF data for consumption by the client.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final byte[] get_buffer()
    throws PDFlibException
    {
	return PDF_get_buffer(p);
    }

    /** Get the text of the last thrown exception or the reason of a failed method call.
        @return  Text containing the description of the most recent error condition.
    */
    public final String get_errmsg()
    {
	return PDF_get_errmsg(p);
    }

    /** Get the number of the last thrown exception or the reason of a failed method call.
        @return  The error code of the most recent error condition.
    */
    public final int get_errnum()
    {
	return PDF_get_errnum(p);
    }

    /** Retrieve some option or other value.
        @return  The value of some option value as requested by keyword.
        @param keyword keyword
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double get_option(String keyword, String optlist)
    throws PDFlibException
    {
	return PDF_get_option(p, keyword, optlist);
    }

    /** Retrieve a string value.
        @return  a string identified by a string index returned by another method.
        @param idx idx
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final String get_string(int idx, String optlist)
    throws PDFlibException
    {
	return PDF_get_string(p, idx, optlist);
    }

    /** Query detailed information about a loaded font.
        @return  The value of some font property as requested by keyword.
        @param font font
        @param keyword keyword
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double info_font(int font, String keyword, String optlist)
    throws PDFlibException
    {
	return PDF_info_font(p, font, keyword, optlist);
    }

    /** Format vector graphics and query metrics and other properties.
        @return  The value of some graphics metrics as requested by keyword.
        @param graphics graphics
        @param keyword keyword
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double info_graphics(int graphics, String keyword, String optlist)
    throws PDFlibException
    {
	return PDF_info_graphics(p, graphics, keyword, optlist);
    }

    /** Format an image and query metrics and other image properties.
        @return  The value of some image metrics as requested by keyword.
        @param image image
        @param keyword keyword
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double info_image(int image, String keyword, String optlist)
    throws PDFlibException
    {
	return PDF_info_image(p, image, keyword, optlist);
    }

    /** Query information about a matchbox on the current page.
        @return  The value of some matchbox parameter as requested by keyword.
        @param boxname boxname
        @param num num
        @param keyword keyword
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double info_matchbox(String boxname, int num, String keyword)
    throws PDFlibException
    {
	return PDF_info_matchbox(p, boxname, num, keyword);
    }

    /** Query the results of drawing a path object without actually drawing it.
        @return  The value of some geometrical values as requested by keyword.
        @param path path
        @param keyword keyword
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double info_path(int path, String keyword, String optlist)
    throws PDFlibException
    {
	return PDF_info_path(p, path, keyword, optlist);
    }

    /** Perform formatting calculations for a PDI page and query the resulting metrics.
        @return  The value of some page metrics as requested by keyword.
        @param page page
        @param keyword keyword
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double info_pdi_page(int page, String keyword, String optlist)
    throws PDFlibException
    {
	return PDF_info_pdi_page(p, page, keyword, optlist);
    }

    /** Query properties of a virtual file or the PDFlib Virtual Filesystem (PVF).
        @return  The value of some file parameter as requested by keyword.
        @param filename filename
        @param keyword keyword
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double info_pvf(String filename, String keyword)
    throws PDFlibException
    {
	return PDF_info_pvf(p, filename, keyword);
    }

    /** Query table information related to the most recently placed table instance.
        @return  The value of some table parameter as requested by keyword.
        @param table table
        @param keyword keyword
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double info_table(int table, String keyword)
    throws PDFlibException
    {
	return PDF_info_table(p, table, keyword);
    }

    /** Query the current state of a Textflow.
        @return  The value of some Textflow parameter as requested by keyword.
        @param textflow textflow
        @param keyword keyword
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double info_textflow(int textflow, String keyword)
    throws PDFlibException
    {
	return PDF_info_textflow(p, textflow, keyword);
    }

    /** Perform textline formatting without creating output and query the resulting metrics.
        @return  The value of some text metric value as requested by keyword.
        @param text text
        @param keyword keyword
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double info_textline(String text, String keyword, String optlist)
    throws PDFlibException
    {
	return PDF_info_textline(p, text, keyword, optlist);
    }

    /** Draw a line from the current point to another point.
        @param x x
        @param y y
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void lineto(double x, double y)
    throws PDFlibException
    {
	PDF_lineto(p, x, y);
    }

    /** Load a 3D model from a disk-based or virtual file.
        @return  A 3D handle, or -1 on error.
        @param filename filename
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int load_3ddata(String filename, String optlist)
    throws PDFlibException
    {
	return PDF_load_3ddata(p, filename, optlist);
    }

    /** Load a multimedia asset or file attachment from a file or URL.
        @return  An asset handle, or -1 on error.
        @param type type
        @param filename filename
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int load_asset(String type, String filename, String optlist)
    throws PDFlibException
    {
	return PDF_load_asset(p, type, filename, optlist);
    }

    /** Search for a font and prepare it for later use.
        @return  A font handle.
        @param fontname fontname
        @param encoding encoding
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int load_font(String fontname, String encoding, String optlist)
    throws PDFlibException
    {
	return PDF_load_font(p, fontname, encoding, optlist);
    }

    /** Open a disk-based or virtual vector graphics file subject to various options.
        @return  A graphics handle, or -1 on error.
        @param type type
        @param filename filename
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int load_graphics(String type, String filename, String optlist)
    throws PDFlibException
    {
	return PDF_load_graphics(p, type, filename, optlist);
    }

    /** Search for an ICC profile, and prepare it for later use.
        @return  A profile handle.
        @param profilename profilename
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int load_iccprofile(String profilename, String optlist)
    throws PDFlibException
    {
	return PDF_load_iccprofile(p, profilename, optlist);
    }

    /** Open a disk-based or virtual image file subject to various options.
        @return  An image handle, or -1 on error.
        @param imagetype imagetype
        @param filename filename
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int load_image(String imagetype, String filename, String optlist)
    throws PDFlibException
    {
	return PDF_load_image(p, imagetype, filename, optlist);
    }

    /** Find a built-in spot color name, or make a named spot color from the current fill color.
        @return  A color handle.
        @param spotname spotname
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int makespotcolor(String spotname)
    throws PDFlibException
    {
	return PDF_makespotcolor(p, spotname);
    }

    /** Add a marked content point with optional properties.
        @param tagname tagname
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void mc_point(String tagname, String optlist)
    throws PDFlibException
    {
	PDF_mc_point(p, tagname, optlist);
    }

    /** Set the current point for graphics output.
        @param x x
        @param y y
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void moveto(double x, double y)
    throws PDFlibException
    {
	PDF_moveto(p, x, y);
    }

    /** Open a disk-based or virtual PDF document and prepare it for later use.
        @return  A PDI document handle.
        @param filename filename
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int open_pdi_document(String filename, String optlist)
    throws PDFlibException
    {
	return PDF_open_pdi_document(p, filename, optlist);
    }

    /** Prepare a page for later use with PDF_fit_pdi_page().
        @return  A page handle.
        @param doc doc
        @param pagenumber pagenumber
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int open_pdi_page(int doc, int pagenumber, String optlist)
    throws PDFlibException
    {
	return PDF_open_pdi_page(p, doc, pagenumber, optlist);
    }

    /** Get the value of a pCOS path with type number or boolean.
        @return  The numerical value of the object identified by the pCOS path.
        @param doc doc
        @param path path
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double pcos_get_number(int doc, String path)
    throws PDFlibException
    {
	return PDF_pcos_get_number(p, doc, path);
    }

    /** Get the value of a pCOS path with type name, number, string, or boolean.
        @return  A string with the value of the object identified by the pCOS path.
        @param doc doc
        @param path path
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final String pcos_get_string(int doc, String path)
    throws PDFlibException
    {
	return PDF_pcos_get_string(p, doc, path);
    }

    /** Get the contents of a pCOS path with type stream, fstream, or string.
        @return  The unencrypted data contained in the stream or string.
        @param doc doc
        @param optlist optlist
        @param path path
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final byte[] pcos_get_stream(int doc, String optlist, String path)
    throws PDFlibException
    {
	return PDF_pcos_get_stream(p, doc, optlist, path);
    }

    /** Delete a PDF container object.
        @param container container
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void poca_delete(int container, String optlist)
    throws PDFlibException
    {
	PDF_poca_delete(p, container, optlist);
    }

    /** Insert a simple or container object in a PDF container object.
        @param container container
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void poca_insert(int container, String optlist)
    throws PDFlibException
    {
	PDF_poca_insert(p, container, optlist);
    }

    /** Create a new PDF container object of type dictionary or array and insert objects.
        @return  A container handle which can be used until it is deleted with PDF_poca_delete().
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int poca_new(String optlist)
    throws PDFlibException
    {
	return PDF_poca_new(p, optlist);
    }

    /** Remove a simple or container object from a PDF container object.
        @param container container
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void poca_remove(int container, String optlist)
    throws PDFlibException
    {
	PDF_poca_remove(p, container, optlist);
    }

    /** Process certain elements of an imported PDF document.
        @return  -1 on error, and 1 otherwise.
        @param doc doc
        @param page page
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int process_pdi(int doc, int page, String optlist)
    throws PDFlibException
    {
	return PDF_process_pdi(p, doc, page, optlist);
    }

    /** Draw a rectangle.
        @param x x
        @param y y
        @param width width
        @param height height
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void rect(double x, double y, double width, double height)
    throws PDFlibException
    {
	PDF_rect(p, x, y, width, height);
    }

    /** Restore the most recently saved graphics state from the stack.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void restore()
    throws PDFlibException
    {
	PDF_restore(p);
    }

    /** Resume a page to add more content to it.
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void resume_page(String optlist)
    throws PDFlibException
    {
	PDF_resume_page(p, optlist);
    }

    /** Rotate the coordinate system.
        @param phi phi
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void rotate(double phi)
    throws PDFlibException
    {
	PDF_rotate(p, phi);
    }

    /** Save the current graphics state to a stack.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void save()
    throws PDFlibException
    {
	PDF_save(p);
    }

    /** Scale the coordinate system.
        @param sx sx
        @param sy sy
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void scale(double sx, double sy)
    throws PDFlibException
    {
	PDF_scale(p, sx, sy);
    }

    /** Set one or more graphics appearance options.
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void set_graphics_option(String optlist)
    throws PDFlibException
    {
	PDF_set_graphics_option(p, optlist);
    }

    /** Activate a graphics state object.
        @param gstate gstate
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void set_gstate(int gstate)
    throws PDFlibException
    {
	PDF_set_gstate(p, gstate);
    }

    /** Fill document information field key with value.
        @param key key
        @param value value
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void set_info(String key, String value)
    throws PDFlibException
    {
	PDF_set_info(p, key, value);
    }

    /** Define layer relationships.
        @param type type
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void set_layer_dependency(String type, String optlist)
    throws PDFlibException
    {
	PDF_set_layer_dependency(p, type, optlist);
    }

    /** Set one or more global options.
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void set_option(String optlist)
    throws PDFlibException
    {
	PDF_set_option(p, optlist);
    }

    /** Set one or more text filter or text appearance options for simple text output methods.
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void set_text_option(String optlist)
    throws PDFlibException
    {
	PDF_set_text_option(p, optlist);
    }

    /** Set the position for simple text output on the page.
        @param x x
        @param y y
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void set_text_pos(double x, double y)
    throws PDFlibException
    {
	PDF_set_text_pos(p, x, y);
    }

    /** Set the color space and color for the graphics and text state..
        @param fstype fstype
        @param colorspace colorspace
        @param c1 c1
        @param c2 c2
        @param c3 c3
        @param c4 c4
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void setcolor(String fstype, String colorspace, double c1, double c2, double c3, double c4)
    throws PDFlibException
    {
	PDF_setcolor(p, fstype, colorspace, c1, c2, c3, c4);
    }

    /** Set the current font in the specified size.
        @param font font
        @param fontsize fontsize
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void setfont(int font, double fontsize)
    throws PDFlibException
    {
	PDF_setfont(p, font, fontsize);
    }

    /** Set the current linewidth.
        @param width width
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void setlinewidth(double width)
    throws PDFlibException
    {
	PDF_setlinewidth(p, width);
    }

    /** Explicitly set the current transformation matrix.
        @param a a
        @param b b
        @param c c
        @param d d
        @param e e
        @param f f
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void setmatrix(double a, double b, double c, double d, double e, double f)
    throws PDFlibException
    {
	PDF_setmatrix(p, a, b, c, d, e, f);
    }

    /** Define a shading (color gradient) between two or more colors.
        @return  A shading handle.
        @param type type
        @param x0 x0
        @param y0 y0
        @param x1 x1
        @param y1 y1
        @param c1 c1
        @param c2 c2
        @param c3 c3
        @param c4 c4
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int shading(String type, double x0, double y0, double x1, double y1, double c1, double c2, double c3, double c4, String optlist)
    throws PDFlibException
    {
	return PDF_shading(p, type, x0, y0, x1, y1, c1, c2, c3, c4, optlist);
    }

    /** Define a shading pattern using a shading object.
        @return  A pattern handle.
        @param shading shading
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final int shading_pattern(int shading, String optlist)
    throws PDFlibException
    {
	return PDF_shading_pattern(p, shading, optlist);
    }

    /** Fill an area with a shading, based on a shading object.
        @param shading shading
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void shfill(int shading)
    throws PDFlibException
    {
	PDF_shfill(p, shading);
    }

    /** Print text in the current font and size at the current position.
        @param text text
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void show(String text)
    throws PDFlibException
    {
	PDF_show(p, text);
    }

    /** Print text in the current font at the specified position.
        @param text text
        @param x x
        @param y y
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void show_xy(String text, double x, double y)
    throws PDFlibException
    {
	PDF_show_xy(p, text, x, y);
    }

    /** Skew the coordinate system.
        @param alpha alpha
        @param beta beta
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void skew(double alpha, double beta)
    throws PDFlibException
    {
	PDF_skew(p, alpha, beta);
    }

    /** Calculate the width of text in an arbitrary font.
        @return  The width of text.
        @param text text
        @param font font
        @param fontsize fontsize
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final double stringwidth(String text, int font, double fontsize)
    throws PDFlibException
    {
	return PDF_stringwidth(p, text, font, fontsize);
    }

    /** Stroke the path with the current color and line width, and clear it.
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void stroke()
    throws PDFlibException
    {
	PDF_stroke(p);
    }

    /** Suspend the current page so that it can later be resumed.
        @param optlist optlist
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void suspend_page(String optlist)
    throws PDFlibException
    {
	PDF_suspend_page(p, optlist);
    }

    /** Translate the origin of the coordinate system.
        @param tx tx
        @param ty ty
        @exception com.pdflib.PDFlibException
	PDF output cannot be finished after an exception.
    */
    public final void translate(double tx, double ty)
    throws PDFlibException
    {
	PDF_translate(p, tx, ty);
    }

    /** Create a new PDFlib object.
        @exception com.pdflib.PDFlibException
        PDF output cannot be finished after an exception.
    */
    public pdflib()
    throws PDFlibException
    {
        p = PDF_new();
    }

    /** Delete a PDFlib object and free all internal resources.
        Never throws any PDFlib exception.
     */
    public final void delete()
    {
        PDF_delete(p);
        p = (long) 0;
    }


    // ------------------------------------------------------------------------
    // IpCOS-specific functions
    
    /**
     * Open a disk-based or virtual PDF document via the IpCOS interface.
     *  
     * @param filename  The full path name of the PDF file to be opened.
     *          The file will be searched by means of the SearchPath resource.
     * @param optlist   An option list specifying document options.
     * 
     *  @exception java.lang.Exception
     *  PDF output cannot be finished after an exception.
     */
    public int pcos_open_document(String filename, String optlist) throws Exception
    {
        return open_pdi_document(filename, optlist);
    }
    
    /**
     * Close PLOP input document via the IpCOS interface.
     * 
     * @param doc       A valid document handle obtained with open_document().
     * @param optlist   An option list specifying document options.
     *
     *  @exception java.lang.Exception
     *  PDF output cannot be finished after an exception.
     */
    public void pcos_close_document(int doc, String optlist) throws Exception
    {
        close_pdi_document(doc);
    }
    
    // private functions

    private long p;

    protected final void finalize()
    throws PDFlibException
    {
	PDF_delete(p);
	p = (long) 0;
    }

    private final static native 
        //AS400_activate synchronized
            long PDF_new() throws PDFlibException;
    private final static native
        //AS400_activate synchronized
            void PDF_delete(long jp);
    private final static native void PDF_activate_item(long jxp, int jid) throws PDFlibException;
    private final static native void PDF_add_nameddest(long jxp, String jname, String joptlist) throws PDFlibException;
    private final static native int PDF_add_path_point(long jxp, int jpath, double jx, double jy, String jtype, String joptlist) throws PDFlibException;
    private final static native int PDF_add_portfolio_file(long jxp, int jfolder, String jfilename, String joptlist) throws PDFlibException;
    private final static native int PDF_add_portfolio_folder(long jxp, int jparent, String jfoldername, String joptlist) throws PDFlibException;
    private final static native int PDF_add_table_cell(long jxp, int jtable, int jcolumn, int jrow, String jtext, String joptlist) throws PDFlibException;
    private final static native int PDF_add_textflow(long jxp, int jtextflow, String jtext, String joptlist) throws PDFlibException;
    private final static native void PDF_align(long jxp, double jdx, double jdy) throws PDFlibException;
    private final static native void PDF_arc(long jxp, double jx, double jy, double jr, double jalpha, double jbeta) throws PDFlibException;
    private final static native void PDF_arcn(long jxp, double jx, double jy, double jr, double jalpha, double jbeta) throws PDFlibException;
    private final static native int PDF_begin_document(long jxp, String jfilename, String joptlist) throws PDFlibException;
    private final static native void PDF_begin_dpart(long jxp, String joptlist) throws PDFlibException;
    private final static native void PDF_begin_font(long jxp, String jfontname, double ja, double jb, double jc, double jd, double je, double jf, String joptlist) throws PDFlibException;
    private final static native void PDF_begin_glyph_ext(long jxp, int juv, String joptlist) throws PDFlibException;
    private final static native int PDF_begin_item(long jxp, String jtagname, String joptlist) throws PDFlibException;
    private final static native void PDF_begin_layer(long jxp, int jlayer) throws PDFlibException;
    private final static native void PDF_begin_mc(long jxp, String jtagname, String joptlist) throws PDFlibException;
    private final static native void PDF_begin_page_ext(long jxp, double jwidth, double jheight, String joptlist) throws PDFlibException;
    private final static native int PDF_begin_pattern_ext(long jxp, double jwidth, double jheight, String joptlist) throws PDFlibException;
    private final static native int PDF_begin_template_ext(long jxp, double jwidth, double jheight, String joptlist) throws PDFlibException;
    private final static native void PDF_circle(long jxp, double jx, double jy, double jr) throws PDFlibException;
    private final static native void PDF_circular_arc(long jxp, double jx1, double jy1, double jx2, double jy2) throws PDFlibException;
    private final static native void PDF_clip(long jxp) throws PDFlibException;
    private final static native void PDF_close_font(long jxp, int jfont) throws PDFlibException;
    private final static native void PDF_close_graphics(long jxp, int jgraphics) throws PDFlibException;
    private final static native void PDF_close_image(long jxp, int jimage) throws PDFlibException;
    private final static native void PDF_close_pdi_document(long jxp, int jdoc) throws PDFlibException;
    private final static native void PDF_close_pdi_page(long jxp, int jpage) throws PDFlibException;
    private final static native void PDF_closepath(long jxp) throws PDFlibException;
    private final static native void PDF_closepath_fill_stroke(long jxp) throws PDFlibException;
    private final static native void PDF_closepath_stroke(long jxp) throws PDFlibException;
    private final static native void PDF_concat(long jxp, double ja, double jb, double jc, double jd, double je, double jf) throws PDFlibException;
    private final static native void PDF_continue_text(long jxp, String jtext) throws PDFlibException;
    private final static native String PDF_convert_to_unicode(long jxp, String jinputformat, byte[] jinputstring, String joptlist) throws PDFlibException;
    private final static native int PDF_create_3dview(long jxp, String jusername, String joptlist) throws PDFlibException;
    private final static native int PDF_create_action(long jxp, String jtype, String joptlist) throws PDFlibException;
    private final static native void PDF_create_annotation(long jxp, double jllx, double jlly, double jurx, double jury, String jtype, String joptlist) throws PDFlibException;
    private final static native int PDF_create_devicen(long jxp, String joptlist) throws PDFlibException;
    private final static native int PDF_create_bookmark(long jxp, String jtext, String joptlist) throws PDFlibException;
    private final static native void PDF_create_field(long jxp, double jllx, double jlly, double jurx, double jury, String jname, String jtype, String joptlist) throws PDFlibException;
    private final static native void PDF_create_fieldgroup(long jxp, String jname, String joptlist) throws PDFlibException;
    private final static native int PDF_create_gstate(long jxp, String joptlist) throws PDFlibException;
    private final static native void PDF_create_pvf(long jxp, String jfilename, byte[] jdata, String joptlist) throws PDFlibException;
    private final static native int PDF_create_textflow(long jxp, String jtext, String joptlist) throws PDFlibException;
    private final static native void PDF_curveto(long jxp, double jx1, double jy1, double jx2, double jy2, double jx3, double jy3) throws PDFlibException;
    private final static native int PDF_define_layer(long jxp, String jname, String joptlist) throws PDFlibException;
    private final static native void PDF_delete_path(long jxp, int jpath) throws PDFlibException;
    private final static native int PDF_delete_pvf(long jxp, String jfilename) throws PDFlibException;
    private final static native void PDF_delete_table(long jxp, int jtable, String joptlist) throws PDFlibException;
    private final static native void PDF_delete_textflow(long jxp, int jtextflow) throws PDFlibException;
    private final static native int PDF_download(long jxp, String jfilename, String joptlist) throws PDFlibException;
    private final static native void PDF_draw_path(long jxp, int jpath, double jx, double jy, String joptlist) throws PDFlibException;
    private final static native void PDF_ellipse(long jxp, double jx, double jy, double jrx, double jry) throws PDFlibException;
    private final static native void PDF_elliptical_arc(long jxp, double jx, double jy, double jrx, double jry, String joptlist) throws PDFlibException;
    private final static native void PDF_encoding_set_char(long jxp, String jencoding, int jslot, String jglyphname, int juv) throws PDFlibException;
    private final static native void PDF_end_document(long jxp, String joptlist) throws PDFlibException;
    private final static native void PDF_end_dpart(long jxp, String joptlist) throws PDFlibException;
    private final static native void PDF_end_font(long jxp) throws PDFlibException;
    private final static native void PDF_end_glyph(long jxp) throws PDFlibException;
    private final static native void PDF_end_item(long jxp, int jid) throws PDFlibException;
    private final static native void PDF_end_layer(long jxp) throws PDFlibException;
    private final static native void PDF_end_mc(long jxp) throws PDFlibException;
    private final static native void PDF_end_page_ext(long jxp, String joptlist) throws PDFlibException;
    private final static native void PDF_end_pattern(long jxp) throws PDFlibException;
    private final static native void PDF_end_template_ext(long jxp, double jwidth, double jheight) throws PDFlibException;
    private final static native void PDF_endpath(long jxp) throws PDFlibException;
    private final static native void PDF_fill(long jxp) throws PDFlibException;
    private final static native int PDF_fill_graphicsblock(long jxp, int jpage, String jblockname, int jgraphics, String joptlist) throws PDFlibException;
    private final static native int PDF_fill_imageblock(long jxp, int jpage, String jblockname, int jimage, String joptlist) throws PDFlibException;
    private final static native int PDF_fill_pdfblock(long jxp, int jpage, String jblockname, int jcontents, String joptlist) throws PDFlibException;
    private final static native void PDF_fill_stroke(long jxp) throws PDFlibException;
    private final static native int PDF_fill_textblock(long jxp, int jpage, String jblockname, String jtext, String joptlist) throws PDFlibException;
    private final static native void PDF_fit_graphics(long jxp, int jgraphics, double jx, double jy, String joptlist) throws PDFlibException;
    private final static native void PDF_fit_image(long jxp, int jimage, double jx, double jy, String joptlist) throws PDFlibException;
    private final static native void PDF_fit_pdi_page(long jxp, int jpage, double jx, double jy, String joptlist) throws PDFlibException;
    private final static native String PDF_fit_table(long jxp, int jtable, double jllx, double jlly, double jurx, double jury, String joptlist) throws PDFlibException;
    private final static native String PDF_fit_textflow(long jxp, int jtextflow, double jllx, double jlly, double jurx, double jury, String joptlist) throws PDFlibException;
    private final static native void PDF_fit_textline(long jxp, String jtext, double jx, double jy, String joptlist) throws PDFlibException;
    private final static native String PDF_get_apiname(long jxp);
    private final static native byte[] PDF_get_buffer(long jxp) throws PDFlibException;
    private final static native String PDF_get_errmsg(long jxp);
    private final static native int PDF_get_errnum(long jxp);
    private final static native double PDF_get_option(long jxp, String jkeyword, String joptlist) throws PDFlibException;
    private final static native String PDF_get_string(long jxp, int jidx, String joptlist) throws PDFlibException;
    private final static native double PDF_info_font(long jxp, int jfont, String jkeyword, String joptlist) throws PDFlibException;
    private final static native double PDF_info_graphics(long jxp, int jgraphics, String jkeyword, String joptlist) throws PDFlibException;
    private final static native double PDF_info_image(long jxp, int jimage, String jkeyword, String joptlist) throws PDFlibException;
    private final static native double PDF_info_matchbox(long jxp, String jboxname, int jnum, String jkeyword) throws PDFlibException;
    private final static native double PDF_info_path(long jxp, int jpath, String jkeyword, String joptlist) throws PDFlibException;
    private final static native double PDF_info_pdi_page(long jxp, int jpage, String jkeyword, String joptlist) throws PDFlibException;
    private final static native double PDF_info_pvf(long jxp, String jfilename, String jkeyword) throws PDFlibException;
    private final static native double PDF_info_table(long jxp, int jtable, String jkeyword) throws PDFlibException;
    private final static native double PDF_info_textflow(long jxp, int jtextflow, String jkeyword) throws PDFlibException;
    private final static native double PDF_info_textline(long jxp, String jtext, String jkeyword, String joptlist) throws PDFlibException;
    private final static native void PDF_lineto(long jxp, double jx, double jy) throws PDFlibException;
    private final static native int PDF_load_3ddata(long jxp, String jfilename, String joptlist) throws PDFlibException;
    private final static native int PDF_load_asset(long jxp, String jtype, String jfilename, String joptlist) throws PDFlibException;
    private final static native int PDF_load_font(long jxp, String jfontname, String jencoding, String joptlist) throws PDFlibException;
    private final static native int PDF_load_graphics(long jxp, String jtype, String jfilename, String joptlist) throws PDFlibException;
    private final static native int PDF_load_iccprofile(long jxp, String jprofilename, String joptlist) throws PDFlibException;
    private final static native int PDF_load_image(long jxp, String jimagetype, String jfilename, String joptlist) throws PDFlibException;
    private final static native int PDF_makespotcolor(long jxp, String jspotname) throws PDFlibException;
    private final static native void PDF_mc_point(long jxp, String jtagname, String joptlist) throws PDFlibException;
    private final static native void PDF_moveto(long jxp, double jx, double jy) throws PDFlibException;
    private final static native int PDF_open_pdi_document(long jxp, String jfilename, String joptlist) throws PDFlibException;
    private final static native int PDF_open_pdi_page(long jxp, int jdoc, int jpagenumber, String joptlist) throws PDFlibException;
    private final static native double PDF_pcos_get_number(long jxp, int jdoc, String jpath) throws PDFlibException;
    private final static native String PDF_pcos_get_string(long jxp, int jdoc, String jpath) throws PDFlibException;
    private final static native byte[] PDF_pcos_get_stream(long jxp, int jdoc, String joptlist, String jpath) throws PDFlibException;
    private final static native void PDF_poca_delete(long jxp, int jcontainer, String joptlist) throws PDFlibException;
    private final static native void PDF_poca_insert(long jxp, int jcontainer, String joptlist) throws PDFlibException;
    private final static native int PDF_poca_new(long jxp, String joptlist) throws PDFlibException;
    private final static native void PDF_poca_remove(long jxp, int jcontainer, String joptlist) throws PDFlibException;
    private final static native int PDF_process_pdi(long jxp, int jdoc, int jpage, String joptlist) throws PDFlibException;
    private final static native void PDF_rect(long jxp, double jx, double jy, double jwidth, double jheight) throws PDFlibException;
    private final static native void PDF_restore(long jxp) throws PDFlibException;
    private final static native void PDF_resume_page(long jxp, String joptlist) throws PDFlibException;
    private final static native void PDF_rotate(long jxp, double jphi) throws PDFlibException;
    private final static native void PDF_save(long jxp) throws PDFlibException;
    private final static native void PDF_scale(long jxp, double jsx, double jsy) throws PDFlibException;
    private final static native void PDF_set_graphics_option(long jxp, String joptlist) throws PDFlibException;
    private final static native void PDF_set_gstate(long jxp, int jgstate) throws PDFlibException;
    private final static native void PDF_set_info(long jxp, String jkey, String jvalue) throws PDFlibException;
    private final static native void PDF_set_layer_dependency(long jxp, String jtype, String joptlist) throws PDFlibException;
    private final static native void PDF_set_option(long jxp, String joptlist) throws PDFlibException;
    private final static native void PDF_set_text_option(long jxp, String joptlist) throws PDFlibException;
    private final static native void PDF_set_text_pos(long jxp, double jx, double jy) throws PDFlibException;
    private final static native void PDF_setcolor(long jxp, String jfstype, String jcolorspace, double jc1, double jc2, double jc3, double jc4) throws PDFlibException;
    private final static native void PDF_setfont(long jxp, int jfont, double jfontsize) throws PDFlibException;
    private final static native void PDF_setlinewidth(long jxp, double jwidth) throws PDFlibException;
    private final static native void PDF_setmatrix(long jxp, double ja, double jb, double jc, double jd, double je, double jf) throws PDFlibException;
    private final static native int PDF_shading(long jxp, String jtype, double jx0, double jy0, double jx1, double jy1, double jc1, double jc2, double jc3, double jc4, String joptlist) throws PDFlibException;
    private final static native int PDF_shading_pattern(long jxp, int jshading, String joptlist) throws PDFlibException;
    private final static native void PDF_shfill(long jxp, int jshading) throws PDFlibException;
    private final static native void PDF_show(long jxp, String jtext) throws PDFlibException;
    private final static native void PDF_show_xy(long jxp, String jtext, double jx, double jy) throws PDFlibException;
    private final static native void PDF_skew(long jxp, double jalpha, double jbeta) throws PDFlibException;
    private final static native double PDF_stringwidth(long jxp, String jtext, int jfont, double jfontsize) throws PDFlibException;
    private final static native void PDF_stroke(long jxp) throws PDFlibException;
    private final static native void PDF_suspend_page(long jxp, String joptlist) throws PDFlibException;
    private final static native void PDF_translate(long jxp, double jtx, double jty) throws PDFlibException;
}
