#!/usr/bin/perl
# 
# Starter sample for fallback fonts
#
# Required data: suitable fonts

use PDFlib::PDFlib;
use strict;

# This is where the data files are. Adjust as necessary. 
use constant searchpath => "../data";
use constant outfile => "starter_fallback.pdf";

use constant {
    llx => 30.0,
    lly => 50.0,
    urx => 800.0,
    ury => 570.0
};

use constant headers => (
    "Use case",
    "Option list for the 'fallbackfonts' option",
    "Base font",
    "With fallback font"
);

# Key names used to make a dictionary for the description of the
# testcase entries
my @testcase_keys = qw(usecase fontname fallbackoptions text);

# Function to create a hash describing each testcase
sub make_testcase_hash {
    my $values = $_[0]; # reference to array
    my %result;
    @result{@testcase_keys} = @{$values};
    return \%result;
}

# The testcases organized as an array of references to hashes
my @testcases = map { make_testcase_hash($_) } (

    [
      "Add enlarged pictogram",
      "NotoSerif-Regular",
      # U+261E = WHITE RIGHT POINTING INDEX
      "{fontname=ZapfDingbats forcechars=U+261E fontsize=150% " .
      "textrise=-15%}",
      "hand symbol: &#x261E;"
    ],
    [
      "Add enlarged symbol glyph",
      "NotoSerif-Regular",
      "{fontname=Symbol forcechars=U+2663 fontsize=125%}",
      "club symbol: &#x2663;"
    ],
    [ # Hebrew characters missing in the font will be pulled from Hebrew
      "Add Hebrew characters to Latin font",
      "NotoSerif-Regular",
      "{fontname=NotoSerifHebrew-Regular}", 
      "Hebrew: &#x05D0;"
    ],
);

my $p = new PDFlib::PDFlib;

eval {
    my $optlist;
    
    $p->set_option("SearchPath={{" . searchpath ."}}");
    $p->set_option("charref=true");
    $p->set_option("glyphcheck=replace");

    # This means that formatting and other errors will raise an
    # exception. This simplifies our sample code, but is not
    # recommended for production code.
    $p->set_option("errorpolicy=exception");

    # Set an output path according to the name of the topic 
    if ($p->begin_document(outfile, "") == -1) {
        printf("Error: %s\n", $p->get_errmsg());
        exit(2);
    }

    $p->set_info("Creator", "PDFlib starter sample");
    $p->set_info("Title", "starter_fallback");

    # Start Page 
    $p->begin_page_ext(0, 0, "width=a4.height height=a4.width");

    my $table = -1;

    # Table header 
    my $col = 1;
    foreach my $header (headers) {
        $optlist =
           "fittextline={fontname=NotoSerif-Regular fontsize=10} " .
           "margin=4";
        $table = $p->add_table_cell($table, $col, 1, $header, $optlist);
        $col += 1;
    }

    # Create fallback samples, one use case per row
    my $row = 2;
    foreach my $testcase (@testcases) {
        $col = 1;

        # Column 1: description of the use case 
        $optlist = 
            "fittextline={fontname=NotoSerif-Regular fontsize=10} " .
            "margin=4";
        $table = $p->add_table_cell($table, $col++, $row,
                                    $testcase->{usecase}, $optlist);

        # Column 2: reproduce option list literally 
        $optlist = 
            "fittextline={fontname=NotoSerif-Regular fontsize=10} " .
            "margin=4";
        $table = $p->add_table_cell($table, $col++, $row,
                                    $testcase->{fallbackoptions}, $optlist);

        # Column 3: text with base font 
        $optlist = sprintf
             "fittextline={fontname=%s fontsize=10 " .
             "replacementchar=? } margin=4",
             $testcase->{fontname};
        $table = $p->add_table_cell($table, $col++, $row,
                                    $testcase->{text}, $optlist);

        # Column 4: text with base font and fallback fonts 
        $optlist = sprintf
             "fittextline={fontname=%s " .
             "fontsize=10 fallbackfonts={%s}} margin=4",
             $testcase->{fontname},
             $testcase->{fallbackoptions};
        $table = $p->add_table_cell($table, $col++, $row,
                                    $testcase->{text}, $optlist);
        
        $row += 1;
    }

    # Place the table 
    $optlist = "header=1 fill={{area=rowodd fillcolor={gray 0.9}}} " .
                "stroke={{line=other}} ";
    my $result = $p->fit_table($table, llx, lly, urx, ury, $optlist);

    if ($result eq "_error")
    {
        printf("Couldn't place table: %s\n", $p->get_errmsg());
        exit(2);
    }

    $p->end_page_ext("");
    $p->end_document("");
};

if ($@) {
    die("$0: PDFlib Exception occurred:\n$@");
}
