#!/usr/bin/perl
# 
# Starter sample for OpenType font features
#
# Demonstrate various typographic OpenType features after checking
# whether a particular feature is supported in a font.
#
# Required data: suitable font with OpenType feature tables
#
# This sample uses a default font which includes various features.
# Depending on the implementation of the features you
# may have to replace the sample text below.
#



use PDFlib::PDFlib;
use strict;

# This is where the data files are. Adjust as necessary.
use constant searchpath => "../data";
use constant outfile => "starter_opentype.pdf";

use constant { 
    llx => 50.0,
    lly => 50.0,
    urx => 800.0,
    ury => 550.0
};

# This font will be used unless another one is specified in the table 
use constant defaulttestfont => "NotoSerif-Regular";

use constant headers => (
    "OpenType feature",
    "Option list",
    "Font name",
    "Raw input (feature disabled)",
    "Feature enabled"
);

# Key names used to make a dictionary for the description of the
# testcase entries
my @testcase_keys = qw(description fontname feature text);

# Function to create a hash describing each testcase
sub make_testcase_hash {
    my $values = $_[0]; # reference to array
    my %result;
    @result{@testcase_keys} = @{$values};
    return \%result;
}

# The testcases organized as an array of references to hashes
my @testcases = map { make_testcase_hash($_) } (
    [
      "ligatures",          # description
      "",                   # fontname
      "liga",               # feature
      "ff fi fl ffi ffl"    # text
    ],
    [
      "discretionary ligatures",
      "",
      "dlig",
      "ch tz"
    ],
    [
      "small capitals",
      "",
      "smcp",
      "PostScript"
    ],
    [
      "ordinals",
      "",
      "ordn",
      "1o 2a 3o"
    ],
    [
      "fractions",
      "",
      "frac",
      "1/2 1/4 3/4"
    ],
    [
      "slashed zero",
      "",
      "zero",
      "0"
    ],
    [
      "historical forms",
      "",
      "hist",
      "s"
    ],
    [
      "proportional figures",
      "",
      "pnum",
      "0123456789"
    ],
    [
      "old-style figures",
      "",
      "onum",
      "0123456789"
    ],
    [
      "lining figures",
      "",
      "lnum",
      "0123456789"
    ],
    [
      "superscript",
      "",
      "sups",
      "123"
    ],
    [
      "subscript",
      "",
      "subs",
      "H2SO4"
    ]
);

my $p = new PDFlib::PDFlib;

eval {
    my $optlist;

    $p->set_option("SearchPath={{" . searchpath ."}}");
    $p->set_option("charref=true");

    # This means that formatting and other errors will raise an
    # exception. This simplifies our sample code, but is not
    # recommended for production code.
    $p->set_option("errorpolicy=exception");

    # Set an output path according to the name of the topic 
    if ($p->begin_document(outfile, "") == -1) {
        printf("Error: %s\n", $p->get_errmsg());
        exit(2);
    }

    $p->set_info("Creator", "PDFlib starter sample");
    $p->set_info("Title", "starter_opentype");

    my $table = -1;

    # Table header 
    my $col = 1;
    foreach my $header (headers) {
        $optlist =
           "fittextline={fontname=NotoSerif-Regular fontsize=12} " .
           "margin=4";
        $table = $p->add_table_cell($table, $col, 1, $header, $optlist);
        $col += 1;
    }

    # Create a table with feature samples, one feature per table row 
    my $row = 2;
    foreach my $testcase (@testcases) {
        # Use the entry in the test table if available, and the
        # default test font otherwise. This way we can easily check
        # a font for all features, as well as insert suitable fonts
        # for individual features.
        my $testfont = 
            $testcase->{fontname} ? $testcase->{fontname} : defaulttestfont;

        $col = 1;

        # Common option list for columns 1-3 
        $optlist = 
            "fittextline={fontname=NotoSerif-Regular fontsize=12} " .
            "margin=4";

        # Column 1: feature description 
        $table = $p->add_table_cell($table, $col++, $row,
                                    $testcase->{description}, $optlist);

        # Column 2: option list 
        my $buf = sprintf "features={%s}", $testcase->{feature};
        $table = $p->add_table_cell($table, $col++, $row, $buf, $optlist);

        # Column 3: font name 
        $table = $p->add_table_cell($table, $col++, $row, $testfont,
                                    $optlist);

        # Column 4: raw input text with  feature disabled 
        $optlist = sprintf
             "fittextline={fontname={%s} fontsize=12} " .
             "margin=4", $testfont;
        $table = $p->add_table_cell($table, $col++, $row,
                                    $testcase->{text}, $optlist);

        # Column 5: text with enabled feature, or warning if the
        # feature is not available in the font
        my $font = $p->load_font($testfont, "unicode", "");

        # Check whether font contains the required feature table 
        $optlist = sprintf "name=%s", $testcase->{feature};
        if ($p->info_font($font, "feature", $optlist) == 1) {
            # feature is available: apply it to the text 
            $optlist = sprintf
                 "fittextline={fontname={%s} fontsize=12 " .
                 "features={%s}} margin=4",
                 $testfont, $testcase->{feature};
            $table = $p->add_table_cell($table, $col++, $row,
                                        $testcase->{text}, $optlist);
        }
        else {
            # feature is not available: emit a warning 
            $optlist = 
                 "fittextline={fontname=NotoSerif-Regular " .
                 "fontsize=12 fillcolor=red} margin=4";
            $table = $p->add_table_cell($table, $col++, $row,
                    "(feature not available in this font)", $optlist);
        }
        
        $row += 1;
    }

    # Loop until all of the table is placed; create new pages
    # as long as more table instances need to be placed.
    my $result;
    do {
        $p->begin_page_ext(0, 0, "width=a4.height height=a4.width");

        # Shade every other row; draw lines for all table cells.
        $optlist = "header=1 fill={{area=rowodd " .
            "fillcolor={gray 0.9}}} stroke={{line=other}} ";

        # Place the table instance
        $result = $p->fit_table($table, llx, lly, urx, ury, $optlist);

        if ($result eq "_error") {
            printf("Couldn't place table: %s\n", $p->get_errmsg());
            exit(2);
        }

        $p->end_page_ext("");
    } while ($result eq "_boxfull");
    $p->end_document("");
};

if ($@) {
    die("$0: PDFlib Exception occurred:\n$@");
}
