#!/usr/bin/env ruby
# 
# Starter sample for fallback fonts
#
# Required data: suitable fonts
#

require 'PDFlib'

# This is where the data files are. Adjust as necessary.
searchpath = "../data"
outfile = "starter_fallback.pdf"

llx = 30
lly = 50
urx = 800
ury = 570

headers = [ "Use case",
    "Option list for the 'fallbackfonts' option", "Base font",
    "With fallback font" ]

testcases = [
    {
        "usecase" => "Add enlarged pictogram",
        "fontname" => "NotoSerif-Regular",
        # pointing hand pictogram
        "fallbackoptions" => "{fontname=ZapfDingbats " +
            "forcechars=U+261E fontsize=150% textrise=-15%}",
        "text" => "hand symbol: &#x261E;"},

    {
        "usecase" => "Add enlarged symbol glyph",
        "fontname" => "NotoSerif-Regular",
        "fallbackoptions" => "{fontname=Symbol " +
            "forcechars=U+2663 fontsize=125%}",
        "text" => "club symbol: &#x2663;"},

    # Hebrew characters missing in the font will be pulled from Hebrew
    # font
    {
        "usecase" => "Add Hebrew characters to Latin font",
        "fontname" => "NotoSerif-Regular",
        "fallbackoptions" => "{fontname=NotoSerifHebrew-Regular}",
        "text" => "Hebrew: &#x05D0;"},
]

begin
  p = PDFlib.new

  p.set_option("SearchPath={{" + searchpath +"}}")
  p.set_option("charref=true")
  p.set_option("glyphcheck=replace")

  #
  # This means that formatting and other errors will raise an
  # exception. This simplifies our sample code, but is not
  # recommended for production code.
  #
  p.set_option("errorpolicy=exception")

  # Set an output path according to the name of the topic
  if (p.begin_document(outfile, "") == -1)
      raise "Error: " + p.get_errmsg()
  end

  p.set_info("Creator", "PDFlib starter sample")
  p.set_info("Title", "starter_fallback")

  # Start Page
  p.begin_page_ext(0, 0, "width=a4.height height=a4.width")

  table = -1
  row = 1

  # Table header
  1.step(headers.length) do |col|
    optlist = "fittextline={fontname=NotoSerif-Regular " +
        "fontsize=10} margin=4"
    table = p.add_table_cell(table, col, row, headers[col-1], optlist)
  end

  row = 2
  # Create fallback samples, one use case per row
  testcases.each do |testcase|
    col = 1

    # Column 1: description of the use case
    optlist = "fittextline={fontname=NotoSerif-Regular " +
        "fontsize=10} margin=4"
    table = p.add_table_cell(table, col, row, testcase["usecase"], optlist)
    col +=1

    # Column 2: reproduce option list literally
    optlist = "fittextline={fontname=NotoSerif-Regular " +
        "fontsize=10} margin=4"
    table = p.add_table_cell(table, col, row, testcase["fallbackoptions"], optlist)
    col +=1

    # Column 3: text with base font
    optlist = "fittextline={fontname=" + testcase["fontname"] +
            " " +
            " fontsize=10 replacementchar=? } margin=4"
    table = p.add_table_cell(table, col, row, testcase["text"], optlist)
    col +=1

    # Column 4: text with base font and fallback fonts
    optlist = "fittextline={fontname=" + testcase["fontname"] +
            " " +
            " fontsize=10 fallbackfonts={" +
            testcase["fallbackoptions"] + "}} margin=4"
    table = p.add_table_cell(table, col, row, testcase["text"], optlist)
    col +=1
    row +=1
  end

  # Place the table
  optlist = "header=1 fill={{area=rowodd " +
          "fillcolor={gray 0.9}}} stroke={{line=other}} "
  result = p.fit_table(table, llx, lly, urx, ury, optlist)

  if (result == "_error")
      raise "Couldn't place table: " + p.get_errmsg()
  end

  p.end_page_ext("")
  p.end_document("")

rescue  PDFlibException => pe
  print "PDFlib exception occurred:\n"
  print "[" + pe.get_errnum.to_s + "] " + pe.get_apiname + \
                  ": " + pe.get_errmsg + "\n"
rescue  Exception => e
  print e.backtrace.join("\n") + "\n" + e.to_s
ensure
  p.delete() if p
end

# vim: sw=2
