#!/usr/bin/env ruby
#
# PDF merge starter:
# Merge pages from multiple PDF documents (including annotations)
#
# Required software: PDFlib+PDI or PDFlib Personalization Server (PPS)
# Required data: PDF documents
#

require 'PDFlib'

# This is where the data files are. Adjust as necessary.
searchpath = "../data"

# By default annotations are also imported. In some cases this
# requires the Noto fonts for creating annotation appearance streams.
fontpath =  "../../resource/font"

pdffiles = [
  "markup_annotations.pdf",
  "pCOS-datasheet.pdf"
]

begin
  p = PDFlib.new

  # This means we must check return values of load_font() etc.
  p.set_option("errorpolicy=return")

  p.set_option("SearchPath={{" + searchpath +"}}")
  p.set_option("SearchPath={{" + fontpath +"}}")

  if (p.begin_document("starter_pdfmerge.pdf", "") == -1)
    raise "Error: " + p.get_errmsg()
  end

  p.set_info("Creator", "PDFlib starter sample")
  p.set_info("Title", "starter_pdfmerge")

  pdffiles.each { |pdffile|
    # Open the input PDF
    indoc = p.open_pdi_document(pdffile, "")
    if (indoc == -1)
      print "Error: " + p.get_errmsg()
      continue
    end

    endpage = p.pcos_get_number(indoc, "length:pages")

    # Loop over all pages of the input document
    1.step(endpage, 1) do |pageno|
      page = p.open_pdi_page(indoc, pageno, "")

      if (page == -1)
	print "Error: " + p.get_errmsg()
	continue
      end
      # Dummy page size; will be adjusted later
      p.begin_page_ext(10, 10, "")

      # Create a bookmark with the file name
      if (pageno == 1)
	p.create_bookmark(pdffile, "")
      end

      # Place the imported page on the output page, and
      # adjust the page size. If the page contains annotations
      # these are also imported.
      p.fit_pdi_page(page, 0, 0, "adjustpage")
      p.close_pdi_page(page)

      p.end_page_ext("")
    end
    p.close_pdi_document(indoc)
  }

  p.end_document("")

rescue  PDFlibException => pe
  print "PDFlib exception occurred:\n"
  print "[" + pe.get_errnum.to_s + "] " + pe.get_apiname + \
		": " + pe.get_errmsg + "\n"
rescue  Exception => e
  print e.backtrace.join("\n") + "\n" + e.to_s
ensure
  p.delete() if p
end

# vim: sw=2
