#!/usr/bin/env ruby
#
# PDF/X-4 starter:
# Create PDF/X-4 conforming output with layers and transparency
#
# The document contains transparent text which is allowed in
# PDF/X-4, but not earlier PDF/X standards.
#
# Required data: font file, image file, ICC output intent profile
#                (see www.pdflib.com for output intent ICC profiles)
#

require 'PDFlib'

# This is where the data files are. Adjust as necessary
searchpath = "../data"
imagefile = "zebra.tif"

begin
  # create a new PDFlib object
  p = PDFlib.new

  # This means we must check return values of load_font() etc.
  p.set_option("errorpolicy=return")

  p.set_option("SearchPath={{" + searchpath +"}}")

  if (p.begin_document("starter_pdfx4.pdf", "pdfx=PDF/X-4") == -1)
      raise "Error: " + p.get_errmsg()
  end

  p.set_info("Creator", "PDFlib starter sample")
  p.set_info("Title", "starter_pdfx4")


  if (p.load_iccprofile("ISOcoated_v2_eci.icc", "usage=outputintent") == -1)
      raise "Error: " + p.get_errmsg() + "\n" +
	  "See www.pdflib.com for output intent ICC profiles.\n"
  end

  # Define the layers; "defaultstate" specifies whether or not
  # the layer is visible when the page is opened.
  #

  layer_english = p.define_layer("English text", "defaultstate=true")
  layer_german  = p.define_layer("German text", "defaultstate=false")
  layer_french  = p.define_layer("French text", "defaultstate=false")

  # Define a radio button relationship for the language layers.
  
  optlist = "group={" + layer_english.to_s + " " + layer_german.to_s + " " +
	    layer_french.to_s + "}"
  p.set_layer_dependency("Radiobtn", optlist);

  layer_image   = p.define_layer("Images", "defaultstate=true")

  p.begin_page_ext(0,0, "width=a4.width height=a4.height")

  font = p.load_font("NotoSerif-Regular", "unicode", "")

  if (font == -1)
      raise "Error: " + p.get_errmsg()
  end

  p.setfont(font, 24)

  p.begin_layer(layer_english)

  p.fit_textline("PDF/X-4 starter sample with layers", 50, 700, "")

  p.begin_layer(layer_german)
  p.fit_textline("PDF/X-4 Starter-Beispiel mit Ebenen", 50, 700, "")

  p.begin_layer(layer_french)
  p.fit_textline("PDF/X-4 Starter exemple avec des calques", 50, 700, "")

  p.begin_layer(layer_image)

  image = p.load_image("auto", imagefile, "")
  if (image == -1)
      raise "Error: " + p.get_errmsg()
  end

  # Place the image on the page
  p.fit_image(image,  0.0,  0.0, "")

  # Place a diagonal stamp across the image area
  width = p.info_image(image, "width", "")
  height = p.info_image(image, "height", "")

  optlist = "boxsize={" + width.to_s + " " + height.to_s + "} font="+ font.to_s +
        " stamp=ll2ur fillcolor={lab 100 0 0} gstate={opacityfill=0.5}"
  p.fit_textline("Zebra", 0, 0, optlist)

  p.close_image(image)

  # Close all layers
  p.end_layer()

  p.end_page_ext("")

  p.end_document("")

rescue  PDFlibException => pe
  print "PDFlib exception occurred:\n"
  print "[" + pe.get_errnum.to_s + "] " + pe.get_apiname + \
                  ": " + pe.get_errmsg + "\n"
rescue  Exception => e
  print e.backtrace.join("\n") + "\n" + e.to_s
ensure
  p.delete() if p
end

# vim: sw=2
