/*---------------------------------------------------------------------------*
 |          Copyright (c) 2015-2022 PDFlib GmbH. All rights reserved.        |
 +---------------------------------------------------------------------------+
 |    This software may not be copied or distributed except as expressly     |
 |    authorized by PDFlib GmbH's general license agreement or a custom      |
 |    license agreement signed by PDFlib GmbH.                               |
 |    For more information about licensing please refer to www.pdflib.com.   |
 *---------------------------------------------------------------------------*/
package com.pdflib;

/**
 * Interface for pCOS functions. This interface is defined for
 * purpose of the pCOS Cookbook only and should not be used directly
 * by client code.
 */
public interface IpCOS {
    /**
     * Get the value of a pCOS path with type number or boolean.
     * @param doc	A valid document handle obtained with pcos_open_document()
     * @param path	A full pCOS path for a numerical or boolean object
     *
     * @return  The numerical value of the object identified by the pCOS path.
     * @exception java.lang.Exception	(see manual)
     */  
    public double pcos_get_number(int doc, String path) throws Exception;
    
    /**
     * Get the value of a pCOS path with type name, number, string, or boolean.
     * @param doc	A valid document handle obtained with pcos_open_document()
     * @param path	A full pCOS path for a string, number, name, or boolean object
     * 
     * @return  A string with the value of the object identified by the pCOS path.
     * @exception java.lang.Exception	(see manual)
     */
    public String pcos_get_string(int doc, String path) throws Exception;
    
    /**
     * Get the contents of a pCOS path with type stream, fstream, or string.
     * @param doc	A valid document handle obtained with pcos_open_document()
     * @param optlist	An option list specifying stream retrieval options
     * @param path	A full pCOS path for a stream or string object
     *
     * @return  The unencrypted data contained in the stream or string.
     * @exception java.lang.Exception	(see manual)
     */
    public byte[] pcos_get_stream(int doc, String optlist, String path) throws Exception;

    /**
     * Open a disk-based or virtual PDF document and prepare it for later use.
     * @param filename	The full path name of the PDF file to be opened.
     * The file will be searched by means of the SearchPath resource.
     * @param optlist	An option list with document options.
     *  
     * @return  A document handle.
     * @exception java.lang.Exception	(see manual)
     */
    public int pcos_open_document(String filename, String optlist) throws Exception;
    
    /**
     * Close the input PDF document.
     * @param doc	A valid document handle obtained with pcos_open_document()
     * @param optlist	An option list
     * 
     * @exception java.lang.Exception	(see manual)
     */
    public void pcos_close_document(int doc, String optlist) throws Exception;

    /**
     * Set one or more global options.
     * @param optlist	An option list specifying global options
     * 
     * @exception java.lang.Exception	(see manual)
     */
    public void set_option(String optlist) throws Exception;
    
    /**
     * Create a named virtual read-only file from data provided in memory.
     * @param filename	The name of the virtual file
     * @param data		Data for the virtual file
     * @param optlist	An option list
     * 
     * @exception java.lang.Exception	(see manual)
     */
    public void create_pvf(String filename, byte[] data, String optlist) throws Exception;
    
    /**
     * Delete a named virtual file and free its data structures (but not the contents).
     * @param filename	The name of the virtual file.
     * 
     * @return  -1 if the virtual file exists but is locked, and 1 otherwise.
     * @exception java.lang.Exception	(see manual)
     */
    public int delete_pvf(String filename) throws Exception;
    
    /** Query properties of a virtual file or the PDFlib Virtual Filesystem (PVF).
     * @param filename	The name of the virtual file. The filename may be empty
     * if keyword=filecount.
     * @param keyword	A keyword for the requested information
     * 
     * @return  The value of some file parameter as requested by keyword.
     * @exception java.lang.Exception	(see manual)
     */
    public double info_pvf(String filename, String keyword) throws Exception;
 
    /**
     * Convert a string in an arbitrary encoding to a Unicode string in various formats.
     * @param inputformat	Unicode text format or encoding name specifying
     * interpretation of the input string
     * @param inputstring	String to be converted to Unicode
     * @param optlist	An option list specifying options for input
     * interpretation and Unicode conversion
     * 
     * @return  The converted Unicode string.
     * @exception java.lang.Exception	(see manual)
     */
    public String convert_to_unicode(String inputformat, byte[] inputstring, String optlist) throws Exception;
    
    /**
     * Delete underlying object.
     */
    public void delete();
  
    /**
     * Get the text of the last thrown exception or the reason of a failed function call.
     * 
     * @return  Text containing the description of the most recent error condition.
     */
    public String get_errmsg();
    
    /**
     * Get the number of the last thrown exception or the reason of a failed function call.
     * 
     * @return  The error code of the most recent error condition.
     */
    public int get_errnum();
    
    /**
     * Get the name of the API function which threw the last exception or failed.
     * 
     * @return  Name of an API function.
     */
    public String get_apiname();
}
