/*
 * PDF text extractor based on PDFlib TET
 *
 * The text is written as UTF-32 in native byte order. The program uses the
 * std::u32string type and UTF-32 string literals that are available in C++11
 * and later.
 *
 * The program assumes that the program arguments are encoded as UTF-8.
 */

#include <iostream>
#include <iomanip>
#include <fstream>
#include <codecvt>

#define TET_CPP_U32STRING
#include "tet.hpp"

/* Figure out whether we're running on an EBCDIC-based machine. */
#if 'A' == 0xC1
#define NATIVE_ENCODING U"ebcdicutf8"
#else
#define NATIVE_ENCODING U"utf8"
#endif

using namespace pdflib;
using namespace std;

namespace
{

/*  global option list */
const u32string globaloptlist =
            U"searchpath={{../data} {../../../resource/cmap}}";

/* document-specific  option list */
const u32string docoptlist = U"";

/* page-specific option list */
const u32string pageoptlist = U"granularity=page";

/* separator to emit after each chunk of text. This depends on the
 * applications needs; for granularity=word a space character may be useful.
 */
const u32string separator = U"\n";

u32string get_u32string(const string& utf8_string);
string get_native_string(const u32string& utf32_string);

} // end anonymous namespace

int main(int argc, char **argv)
{
    ofstream out;
    int pageno = 0;
    
    try
    {
        TET tet;

        if (argc != 3)
        {
            cerr << "usage: extractor_u32string <infilename> <outfilename>" << endl;
            return 2;
        }

        out.open(argv[2], ios::binary);
        if (!out.is_open())
        {
            cerr << "Couldn't open output file " << argv[2] << endl;
            return 2;
        }

        u32string::value_type const bom = 0xfeff;
        out.write(reinterpret_cast<char const *>(&bom), sizeof(u32string::value_type));

        tet.set_option(globaloptlist);

        /*
         * Caution: For simplicity we assume that the program arguments are
         * encoded as UTF-8, which might not be true in all cases!
         */
        u32string const doc_name(get_u32string(argv[1]));
        const int doc = tet.open_document(doc_name, docoptlist);

        if (doc == -1)
        {
            cerr << "Error " << tet.get_errnum()
                << " in " << get_native_string(tet.get_apiname()) << "(): "
                << get_native_string(tet.get_errmsg()) << endl;
            return 2;
        }

        /* get number of pages in the document */
        const int n_pages = (int) tet.pcos_get_number(doc, U"length:pages");

        /* loop over pages in the document */
        for (pageno = 1; pageno <= n_pages; ++pageno)
        {
            u32string text;
            const int page = tet.open_page(doc, pageno, pageoptlist);

            if (page == -1)
            {
                cerr << "Error " << tet.get_errnum()
                    << " in " << get_native_string(tet.get_apiname())
                    << "(): " << get_native_string(tet.get_errmsg()) << endl;
                continue;                        /* try next page */
            }

            /* Retrieve all text fragments; This is actually not required
             * for granularity=page, but must be used for other granularities.
             */
            while ((text = tet.get_text(page)) != U"")
            {
                /* print the retrieved text as UTF-32-encoded in the native
                 * byte order
                 */
                out.write(reinterpret_cast<const char *>(text.c_str()),
                        static_cast<streamsize>(text.size())
                            * sizeof(u32string::value_type));

                /* print a separator between chunks of text */
                out.write(reinterpret_cast<const char *>(separator.c_str()),
                        static_cast<streamsize>(separator.size())
                            * sizeof(u32string::value_type));
            }

            if (tet.get_errnum() != 0)
            {
                cerr << "Error " << tet.get_errnum()
                    << " in " << get_native_string(tet.get_apiname())
                    << "() on page " << pageno
                    << ": " << get_native_string(tet.get_errmsg()) << endl;
            }

            tet.close_page(page);
        }

        tet.close_document(doc);
    }
    catch (TET::Exception &ex) {
        if (pageno == 0)
        {
            cerr << "Error " << ex.get_errnum()
                << " in " << get_native_string(ex.get_apiname())
                << "(): " << get_native_string(ex.get_errmsg()) << endl;
        }
        else
        {
            cerr << "Error " << ex.get_errnum()
                << " in " << get_native_string(ex.get_apiname())
                << "() on page " << pageno
                << ": " << get_native_string(ex.get_errmsg()) << endl;
        }
        return 2;
    }
    catch (exception &e) {
        cerr << "C++ exception occurred: " << e.what() << endl;
        return 99;
    }
    catch (...) {
        cerr << "Generic C++ exception occurred!" << endl;
        return 99;
    }

    out.close();
    return 0;
}

namespace
{

// This TET object is used solely for conversion purposes between the encoding
// of the TET API parameters and the native encoding.
TET conversion_helper;

u32string get_u32string(const string& native_string)
{
    string const u32_bytes =
        conversion_helper.convert_to_unicode(NATIVE_ENCODING, native_string,
                                                    U"outputformat=utf32");

    return u32string(
            reinterpret_cast<u32string::value_type const *>(u32_bytes.c_str()),
            u32_bytes.length() / sizeof(u32string::value_type));
}

string get_native_string(const u32string& utf32_string)
{
    string const u32_bytes(
            reinterpret_cast<string::value_type const *>(utf32_string.c_str()),
            utf32_string.length() * sizeof(u32string::value_type));

    return conversion_helper.convert_to_unicode(U"utf32", u32_bytes,
                                            U"outputformat=" NATIVE_ENCODING);
}

} // end anonymous namespace
